/*
 * unsigned long get_blocks(int fd)
 *
 * 	returns the number of 512-byte blocks of fd
 *
 * Most code ripped from:
 * 
 * 	mkswap.c
 * 	mkfs.bfs.c
 * 	mkfs.minix.c
 *
 * by Guillem Jover <guillem.jover@menta.net>
 * 
 */

#include "defines.h"
#include "get_blocks.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdint.h>
#include <unistd.h>

#ifdef __linux__
#include <sys/ioctl.h>

/* can't #include <linux/fs.h>, because it uses u64... */
#ifndef BLKGETSIZE
/* return device size */
#ifndef _IO
/* pre-1.3.45 */
#define BLKGETSIZE 0x1260
#else
/* same on i386, m68k, arm; different on alpha, mips, sparc, ppc */
#define BLKGETSIZE _IO(0x12,96)

#undef _IOR
#undef _IOW
#define _IOR(type,nr,size)      _IOC(_IOC_READ,(type),(nr),sizeof(size))
#define _IOW(type,nr,size)	_IOC(_IOC_WRITE,(type),(nr),sizeof(size))
#define BLKGETSIZE64 _IOR(0x12,114,sizeof(uint64_t))

#endif
#endif
#endif /* __linux__ */

static int
valid_offset (int fd, off_t offset)
{
	char	ch;

	if (lseek (fd, offset, 0) < 0)
		return 0;
	if (read (fd, &ch, 1) < 1)
		return 0;
	return 1;
}

static off_t 
count_blocks (int fd)
{
	off_t	high, low, blocks;

	low = 0;
	for (high = 1; high > 0 && valid_offset (fd, high); high *= 2)
		low = high;
	while (low < high - 1)
	{
		const off_t	mid = (low + high) / 2;

		if (valid_offset (fd, mid))
			low = mid;
		else
			high = mid;
	}
	blocks = (low + 1) / 512;
	return blocks;
}

unsigned long
get_blocks (int fd)
{
	struct stat	st;

#ifdef __linux__
	{
		unsigned long	blocks;
		unsigned long long bytes;
		if (ioctl(fd, BLKGETSIZE64, &bytes) == 0) {
			blocks = (bytes >> 9);
			return blocks;
		} else {
			blocks = 0;

			if (ioctl(fd, BLKGETSIZE, &blocks) >=0)
				return blocks;
		}
	}
#endif /*  __linux__ */

	if (fstat(fd, &st) == 0)
		return st.st_size / 512;

	return count_blocks(fd);
}

