#include "config.h"
#include "GraphicsSurface.h"

#if USE(GRAPHICS_SURFACE)

#include "NotImplemented.h"
#include "TextureMapperGL.h"

#if PLATFORM(QT)
// Qt headers must be included before glx headers.
#include <QGuiApplication>
#include <QOpenGLContext>
#include <qpa/qplatformnativeinterface.h>
#include <EGL/egl.h>
#include <EGL/eglext.h>
#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include <libgdl.h>
#include <libgma.h>
#include <x86_cache.h>
#endif

namespace WebCore {

static PFNEGLCREATEIMAGEKHRPROC pEglCreateImageKHR;
static PFNEGLDESTROYIMAGEKHRPROC pEglDestroyImageKHR;
static PFNGLEGLIMAGETARGETTEXTURE2DOESPROC pGlEGLImageTargetTexture2D;

struct GraphicsSurfaceBuffer {
    GraphicsSurfaceBuffer(uint32_t id)
        : m_data(0)
        , m_dirty(false)
        , m_texture_id(0)
        , m_egl_image(EGL_NO_IMAGE_KHR)
        , m_egl_display(EGL_NO_DISPLAY)
    {
        if (id == GDL_SURFACE_INVALID) {
            m_surface_info.id = GDL_SURFACE_INVALID;
            return;
        }

        if (gdl_get_surface_info(id, &m_surface_info) != GDL_SUCCESS) {
            qWarning("buffer %u not found", id);
            m_surface_info.id = GDL_SURFACE_INVALID;
        }
    }

    GraphicsSurfaceBuffer(const IntSize &size)
        : m_data(0)
        , m_dirty(false)
        , m_texture_id(0)
        , m_egl_image(EGL_NO_IMAGE_KHR)
        , m_egl_display(EGL_NO_DISPLAY)
    {
        if (gdl_alloc_surface(GDL_PF_ABGR_32, size.width(), size.height(),
                    GDL_SURFACE_CACHED, &m_surface_info) != GDL_SUCCESS) {
            qWarning("failed to allocate %dx%d buffer", size.width(), size.height());
            m_surface_info.id = GDL_SURFACE_INVALID;
        }
    }

    ~GraphicsSurfaceBuffer()
    {
        if (m_surface_info.id != GDL_SURFACE_INVALID) {
            if (m_texture_id)
                glDeleteTextures(1, &m_texture_id);

            if (m_egl_image)
                pEglDestroyImageKHR(eglDisplay(), m_egl_image);

            if (m_data)
                gdl_unmap_surface(m_surface_info.id);

            gdl_free_surface(m_surface_info.id);
        }
    }

    char *lock(int *stride)
    {
        if (!m_data && m_surface_info.id != GDL_SURFACE_INVALID)
            gdl_map_surface(m_surface_info.id, (gdl_uint8**)&m_data, 0);

        if (m_data && stride)
            *stride = m_surface_info.pitch;

        return m_data;
    }

    void unlock()
    {
        if (m_data)
            m_dirty = true;
    }

    EGLDisplay eglDisplay()
    {
        if (m_egl_display == EGL_NO_DISPLAY)
            m_egl_display = QGuiApplication::platformNativeInterface()->nativeResourceForIntegration("eglDisplay");
        return m_egl_display;
    }

    uint32_t id() const
    {
        return m_surface_info.id;
    }

    uint32_t textureID()
    {
        if (m_egl_image == EGL_NO_IMAGE_KHR) {
            if (!pEglCreateImageKHR)
                pEglCreateImageKHR = (PFNEGLCREATEIMAGEKHRPROC)
                    eglGetProcAddress("eglCreateImageKHR");

            if (!pEglDestroyImageKHR)
                pEglDestroyImageKHR = (PFNEGLDESTROYIMAGEKHRPROC)
                    eglGetProcAddress("eglDestroyImageKHR");

            if (!pGlEGLImageTargetTexture2D)
                pGlEGLImageTargetTexture2D = (PFNGLEGLIMAGETARGETTEXTURE2DOESPROC)
                    eglGetProcAddress("glEGLImageTargetTexture2DOES");

            gma_pixmap_info_t pixmap_info;
            pixmap_info.type = GMA_PIXMAP_TYPE_PHYSICAL;
            pixmap_info.virt_addr = lock(NULL);
            pixmap_info.phys_addr = m_surface_info.phys_addr;
            pixmap_info.format = GMA_PF_ABGR_32;
            pixmap_info.width = m_surface_info.width;
            pixmap_info.height = m_surface_info.height;
            pixmap_info.pitch = m_surface_info.pitch;
            pixmap_info.user_data = 0;

            gma_pixmap_t pixmap;
            if (gma_pixmap_alloc(&pixmap_info, NULL, &pixmap) != GMA_SUCCESS) {
                qWarning("intelce: failed to allocate gma pixmap");
                return 0;
            }

            m_egl_image = pEglCreateImageKHR(eglDisplay(), EGL_NO_CONTEXT,
                    EGL_NATIVE_PIXMAP_KHR, pixmap, NULL);

            gma_pixmap_release(&pixmap);

            if (m_egl_image == EGL_NO_IMAGE_KHR)
                return 0;

            if (m_texture_id == 0)
                glGenTextures(1, &m_texture_id);

            GLint prevTexture;
            glGetIntegerv(GL_TEXTURE_2D, &prevTexture);

            glBindTexture(GL_TEXTURE_2D, m_texture_id);
            glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
            glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
            pGlEGLImageTargetTexture2D(GL_TEXTURE_2D, m_egl_image);
            glBindTexture(GL_TEXTURE_2D, prevTexture);
        }

        if (m_dirty) {
            cache_flush_buffer(m_data,
                    m_surface_info.pitch * m_surface_info.height);
            m_dirty = false;
        }

        return m_texture_id;
    }

private:
    gdl_surface_info_t m_surface_info;
    char *m_data;
    bool m_dirty;
    uint32_t m_texture_id;
    EGLImageKHR m_egl_image;
    EGLDisplay m_egl_display;
};

struct GraphicsSurfacePrivate {
public:
    GraphicsSurfacePrivate(const GraphicsSurfaceToken& token, const IntSize& size)
        : m_size(size)
        , m_front(0)
        , m_token(token)
    {
        m_buffers[0] = new GraphicsSurfaceBuffer(token.frontBufferHandle);

        if (token.backBufferHandle != GDL_SURFACE_INVALID)
            m_buffers[1] = new GraphicsSurfaceBuffer(token.backBufferHandle);
        else
            m_buffers[1] = 0;
    }

    GraphicsSurfacePrivate(const PlatformGraphicsContext3D shareContext, const IntSize& size, GraphicsSurface::Flags flags)
        : m_size(size)
        , m_front(0)
    {
        m_buffers[0] = new GraphicsSurfaceBuffer(size);
        if (!(flags & GraphicsSurface::SupportsSingleBuffered))
            m_buffers[1] = new GraphicsSurfaceBuffer(size);
        else
            m_buffers[1] = 0;

        if (!(flags & GraphicsSurface::SupportsSharing))
            return;

        uint32_t front = m_buffers[0]->id();
        uint32_t back = m_buffers[1] ? m_buffers[1]->id() : GDL_SURFACE_INVALID;

        m_token = GraphicsSurfaceToken(front, back);
    }

    ~GraphicsSurfacePrivate()
    {
        delete m_buffers[0];
        delete m_buffers[1];
    }

    GraphicsSurfaceToken token() const
    {
        return m_token;
    }

    GraphicsSurfaceBuffer *frontBuffer()
    {
        return m_buffers[m_front];
    }

    GraphicsSurfaceBuffer *backBuffer()
    {
        return m_buffers[!m_front];
    }

    IntSize size() const
    {
        return m_size;
    }

    void swapBuffers()
    {
        if (m_buffers[!m_front])
            m_front = !m_front;
    }

private:
    IntSize m_size;
    int m_front;
    GraphicsSurfaceBuffer *m_buffers[2];
    GraphicsSurfaceToken m_token;
};


GraphicsSurfaceToken GraphicsSurface::platformExport()
{
    GraphicsSurfaceToken t = m_private->token();
    return t;
}

uint32_t GraphicsSurface::platformGetTextureID()
{
    return m_private->frontBuffer()->textureID();
}

void GraphicsSurface::platformCopyToGLTexture(uint32_t target, uint32_t texture, const IntRect& targetRect, const IntPoint& offset)
{
    if (!m_fbo)
        glGenFramebuffers(1, &m_fbo);

    glBindTexture(GL_TEXTURE_2D, 0);
    glBindTexture(target, texture);
    glBindFramebuffer(GL_FRAMEBUFFER, m_fbo);
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, m_private->frontBuffer()->textureID(), 0);
    glCopyTexSubImage2D(target, 0, targetRect.x(), targetRect.y(), offset.x(), offset.y(), targetRect.width(), targetRect.height());
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, 0, 0);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

void GraphicsSurface::platformCopyFromTexture(uint32_t texture, const IntRect& sourceRect)
{
    GLint previousFBO;
    glGetIntegerv(GL_FRAMEBUFFER_BINDING, &previousFBO);

    GLint prevTexture;
    glGetIntegerv(GL_TEXTURE_2D, &prevTexture);

    GLuint originFBO;
    glGenFramebuffers(1, &originFBO);
    glBindFramebuffer(GL_FRAMEBUFFER, originFBO);
    glBindTexture(GL_TEXTURE_2D, texture);
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, texture, 0);

    glBindTexture(GL_TEXTURE_2D, m_private->backBuffer()->textureID());
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, sourceRect.x(), sourceRect.y(), sourceRect.x(), sourceRect.y(), sourceRect.width(), sourceRect.height());
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, 0, 0);

    glBindTexture(GL_TEXTURE_2D, prevTexture);
    glBindFramebuffer(GL_FRAMEBUFFER, previousFBO);

    glDeleteFramebuffers(1, &originFBO);
    swapBuffers();
}

void GraphicsSurface::platformPaintToTextureMapper(TextureMapper* textureMapper, const FloatRect& targetRect, const TransformationMatrix& transform, float opacity)
{
    IntSize size = m_private->size();
    FloatRect rectOnContents(FloatPoint::zero(), size);
    TransformationMatrix adjustedTransform = transform;
    adjustedTransform.multiply(TransformationMatrix::rectToRect(rectOnContents, targetRect));
    static_cast<TextureMapperGL*>(textureMapper)->drawTexture(m_private->frontBuffer()->textureID(), TextureMapperGL::ShouldBlend | TextureMapperGL::ShouldFlipTexture, size, rectOnContents, adjustedTransform, opacity);
}

uint32_t GraphicsSurface::platformFrontBuffer() const
{
    return m_private->frontBuffer()->id();
}

uint32_t GraphicsSurface::platformSwapBuffers()
{
    m_private->swapBuffers();
    return 0;
}

IntSize GraphicsSurface::platformSize() const
{
    return m_private->size();
}

PassRefPtr<GraphicsSurface> GraphicsSurface::platformCreate(const IntSize& size, Flags flags, const PlatformGraphicsContext3D shareContext)
{
    if (flags & SupportsCopyToTexture || flags & SupportsSingleBuffered)
        return PassRefPtr<GraphicsSurface>();

    RefPtr<GraphicsSurface> surface = adoptRef(new GraphicsSurface(size, flags));
    surface->m_private = new GraphicsSurfacePrivate(shareContext, size, flags);

    return surface;
}

PassRefPtr<GraphicsSurface> GraphicsSurface::platformImport(const IntSize& size, Flags flags, const GraphicsSurfaceToken& token)
{
    RefPtr<GraphicsSurface> surface = adoptRef(new GraphicsSurface(size, flags));
    surface->m_private = new GraphicsSurfacePrivate(token, size);

    return surface;
}

char* GraphicsSurface::platformLock(const IntRect& rect, int* outputStride, LockOptions)
{
    int stride;
    char *data = m_private->frontBuffer()->lock(&stride);

    if (outputStride)
        *outputStride = stride;

    if (data)
        data += rect.y() * stride + rect.x() * 4;

    return data;
}

void GraphicsSurface::platformUnlock()
{
    m_private->frontBuffer()->unlock();
}

void GraphicsSurface::platformDestroy()
{
    if (m_fbo)
        glDeleteFramebuffers(1, &m_fbo);
    if (m_private)
        delete m_private;
    m_private = 0;
}

}
#endif
