/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2010 University of Szeged
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#if PLUGIN_ARCHITECTURE(TPI) && ENABLE(NETSCAPE_PLUGIN_API)

#include "NetscapePlugin.h"

#include "PluginController.h"
#include "WebEvent.h"
#include <WebCore/GraphicsContext.h>
#include <WebCore/NotImplemented.h>

#include <AdobePluginEvents.h>

using namespace WebCore;

class PluginGLFuncs {
private:
    static void *createSurface(int width, int height, int depth)
    {
        if (depth != 32)
            return NULL;
        return new QImage(width, height, QImage::Format_ARGB32_Premultiplied);
    }

    static bool freeSurface(void *surface)
    {
        QImage *image = static_cast<QImage *>(surface);
        delete image;
        return true;
    }

    static bool lockSurface(void *surface)
    {
        return true;
    }

    static bool unlockSurface(void *surface)
    {
        return true;
    }

    static void *getPixels(void *surface)
    {
        QImage *image = static_cast<QImage *>(surface);
        return (void *)image->bits();
    }

    static int getPitch(void *surface)
    {
        QImage *image = static_cast<QImage *>(surface);
        return image->bytesPerLine();
    }

    static bool drawRect(void *surface, uint32_t color, const PluginRect *rect)
    {
        return false;
    }

    static bool blit(void *srcSurface, void *destSurface, int destPitch,
            const PluginRect *srcRect, const PluginRect *destRect,
            bool blendAlpha)
    {
        QImage *image = static_cast<QImage *>(srcSurface);
        QPainter *painter = static_cast<QPainter *>(destSurface);

        QRect srect(srcRect->left, srcRect->top,
                srcRect->right - srcRect->left,
                srcRect->bottom - srcRect->top);

        QRect drect(destRect->left, destRect->top,
                destRect->right - destRect->left,
                destRect->bottom - destRect->top);

        painter->setCompositionMode(blendAlpha ?
                QPainter::CompositionMode_SourceOver :
                QPainter::CompositionMode_Source);

        painter->drawImage(drect, *image, srect);

        return true;
    }

    GLFuncTable m_funcs;
    bool m_enabled;

public:
    PluginGLFuncs() : m_enabled(false) {
        m_funcs.pfCreateSurface = createSurface;
        m_funcs.pfFreeSurface   = freeSurface;
        m_funcs.pfLockSurface   = lockSurface;
        m_funcs.pfUnlockSurface = unlockSurface;
        m_funcs.pfGetPixels     = getPixels;
        m_funcs.pfGetPitch      = getPitch;
        m_funcs.pfDrawRect      = drawRect;
        m_funcs.pfBlit          = blit;
    }

    bool enabled() const { return m_enabled; }
    void setEnabled(bool enabled) { m_enabled = enabled; }
    GLFuncTable *funcs() { return &m_funcs; }
};

static PluginGLFuncs GLFuncs;

namespace WebKit {

void NetscapePlugin::platformPreInitialize()
{
}

bool NetscapePlugin::platformPostInitialize()
{
    // Set the Browser GL Drawing Funcs
    if (!getenv("PLUGIN_DISABLE_GL_FUNCS")) {
        if (NPP_SetValue((NPNVariable)NPPVpluginDrawingFunctions, GLFuncs.funcs()) == NPERR_NO_ERROR)
            GLFuncs.setEnabled(true);
    }

    if (!GLFuncs.enabled())
        qDebug() << "not using plugin rendering methods";

    return true;
}

void NetscapePlugin::platformDestroy()
{
}

bool NetscapePlugin::platformInvalidate(const IntRect&)
{
    notImplemented();
    return false;
}

void NetscapePlugin::platformGeometryDidChange()
{
    if (!GLFuncs.enabled()) {
        QSize size(m_pluginSize.width(), m_pluginSize.height());
        m_image = QImage(size, QImage::Format_RGB32);
    }
}

void NetscapePlugin::platformVisibilityDidChange()
{
    notImplemented();
}

void NetscapePlugin::platformPaint(GraphicsContext* context, const IntRect& dirtyRect, bool /*isSnapshot*/)
{
    PluginEvent evt;

    if (m_isWindowed)
        return;

    if (!m_isStarted) {
        // FIXME: we should paint a missing plugin icon.
        return;
    }

    if (context->paintingDisabled())
        return;

    IntRect exposedRect(dirtyRect);
    QPainter* painter = context->platformContext();

    memset(&evt, 0, sizeof(evt));
    evt.type = PluginPaintEventType;

    evt.paint.bUseAccelBlit = GLFuncs.enabled();
    if (!evt.paint.bUseAccelBlit) {
        evt.paint.pixels = (uint8_t *)m_image.bits();
        evt.paint.pitch = m_image.bytesPerLine();
        evt.paint.depth = m_image.depth() / 8;
    } else
        evt.paint.pixels = (uint8_t *)painter;

    evt.paint.w = m_pluginSize.width();
    evt.paint.h = m_pluginSize.height();

    evt.paint.inval.left = exposedRect.x();
    evt.paint.inval.top = exposedRect.y();
    evt.paint.inval.right = exposedRect.x() + exposedRect.width();
    evt.paint.inval.bottom = exposedRect.y() + exposedRect.height();

    NPP_HandleEvent(&evt);

    if (!evt.paint.bUseAccelBlit)
        painter->drawImage(QPoint(exposedRect.x(), exposedRect.y()), m_image, exposedRect);
}

bool NetscapePlugin::platformHandleMouseEvent(const WebMouseEvent& event)
{
    if (m_isWindowed)
        return false;

    PluginEvent evt;
    memset(&evt, 0, sizeof (PluginEvent));
    evt.type = PluginMouseEventType;

    switch (event.type()) {
    case WebEvent::MouseDown:
        evt.mouse.state = PluginMouseButtonDown;
        break;
    case WebEvent::MouseUp:
        evt.mouse.state = PluginMouseButtonUp;
        break;
    case WebEvent::MouseMove:
        evt.mouse.state = PluginMouseButtonMove;
        break;
    default:
        return false;
    }

    switch (event.button()) {
    case WebMouseEvent::LeftButton:
        evt.mouse.button = PluginMouseButtonLeft;
        break;
    case WebMouseEvent::MiddleButton:
        evt.mouse.button = PluginMouseButtonCenter;
        break;
    case WebMouseEvent::RightButton:
        evt.mouse.button = PluginMouseButtonRight;
        break;
    default:
        evt.mouse.button = -1;
        break;
    }

    IntPoint pluginLocation = convertToRootView(IntPoint());
    evt.mouse.x = event.position().x() - pluginLocation.x();
    evt.mouse.y = event.position().y() - pluginLocation.y();

    return !NPP_HandleEvent(&evt);
}

bool NetscapePlugin::platformHandleWheelEvent(const WebWheelEvent& event)
{
    if (m_isWindowed)
        return false;

    PluginEvent evt;
    memset(&evt, 0, sizeof (PluginEvent));
    evt.type = PluginMouseEventType;
    evt.mouse.state = PluginMouseButtonDown;

    FloatSize ticks = event.wheelTicks();
    if (!ticks.width())
        return false;

    if (ticks.width() > 0)
        evt.mouse.button = PluginMouseButtonWheelDown;
    else
        evt.mouse.button = PluginMouseButtonWheelUp;

    IntPoint pluginLocation = convertToRootView(IntPoint());
    evt.mouse.x = event.position().x() - pluginLocation.x();
    evt.mouse.y = event.position().y() - pluginLocation.y();

    return !NPP_HandleEvent(&evt);
}

void NetscapePlugin::platformSetFocus(bool focusIn)
{
}

bool NetscapePlugin::wantsPluginRelativeNPWindowCoordinates()
{
    return false;
}

bool NetscapePlugin::platformHandleMouseEnterEvent(const WebMouseEvent& event)
{
    return platformHandleMouseEvent(event);
}

bool NetscapePlugin::platformHandleMouseLeaveEvent(const WebMouseEvent& event)
{
    return platformHandleMouseEvent(event);
}

bool NetscapePlugin::platformHandleKeyboardEvent(const WebKeyboardEvent& event)
{
    PluginEvent evt;
    memset(&evt, 0, sizeof (PluginEvent));
    evt.type = PluginKeyEventType;

    switch (event.type()) {
    case WebEvent::KeyDown:
        evt.key.state = PluginKeyDown;
        break;
    case WebEvent::KeyUp:
        evt.key.state = PluginKeyUp;
        break;
    default:
        return false;
    }

    evt.key.scancode = event.nativeVirtualKeyCode();
    evt.key.sym = event.windowsVirtualKeyCode();
    evt.key.unicode = event.text()[0];

    evt.key.mod = 0;
    if (event.shiftKey())
        evt.key.mod |= PluginShiftKey;
    if (event.controlKey())
        evt.key.mod |= PluginCtrlKey;
    if (event.altKey())
        evt.key.mod |= PluginAltKey;
    if (event.metaKey())
        evt.key.mod |= PluginMetaKey;
    if (event.isKeypad())
        evt.key.mod |= PluginNumPad;

    return !NPP_HandleEvent(&evt);
}

} // namespace WebKit

#endif // PLUGIN_ARCHITECTURE(TPI) && ENABLE(NETSCAPE_PLUGIN_API)
