/*
 *  Copyright (C) 2010 Igalia S.L
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "GStreamerGWorld.h"

#if ENABLE(VIDEO) && USE(GSTREAMER)

#include "GRefPtrGStreamer.h"
#include "GStreamerVersioning.h"
#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>
#include <gst/video/videooverlay.h>
#include <gst/wayland/wayland.h>

#include <QtGui/QGuiApplication>
#include <QtGui/qpa/qplatformnativeinterface.h>

using namespace std;

namespace WebCore {

GstBusSyncReply gstGWorldSyncMessageCallback(GstBus*, GstMessage* message, gpointer data)
{
    GStreamerGWorld* gstGWorld = static_cast<GStreamerGWorld*>(data);

    if (gst_is_video_overlay_prepare_window_handle_message(message)) {
        gstGWorld->setWindowOverlay(message);
        return GST_BUS_DROP;
    }

    return GST_BUS_PASS;
}

PassRefPtr<GStreamerGWorld> GStreamerGWorld::createGWorld(GstElement* pipeline)
{
    return adoptRef(new GStreamerGWorld(pipeline));
}

void
registry_handle_global(void *data, struct wl_registry *registry,
        uint32_t id, const char *interface, uint32_t version)
{
    GStreamerGWorld *gstGWorld = static_cast<GStreamerGWorld*>(data);

    if (!strcmp(interface, "wl_compositor")) {
        gstGWorld->wl_compositor = (struct wl_compositor *)
            wl_registry_bind(registry, id, &wl_compositor_interface, 3);

    } else if (!strcmp(interface, "fbx_foreign_surface_manager")) {
        gstGWorld->fbx_fsm = (struct fbx_foreign_surface_manager *)
            wl_registry_bind(registry, id,
                    &fbx_foreign_surface_manager_interface, 1);
    }
}

static void
registry_handle_global_remove(void *data, struct wl_registry *registry,
        uint32_t name)
{
}

static const struct wl_registry_listener wl_registry_listener = {
    registry_handle_global,
    registry_handle_global_remove,
};

static void
foreign_surface_id(void *data, struct fbx_foreign_surface *foreign_surface,
        uint32_t id)
{
    GStreamerGWorld *gstGWorld = static_cast<GStreamerGWorld*>(data);

    gstGWorld->m_overlay_id = id;
}

static void
foreign_surface_imported(void *data, struct fbx_foreign_surface *foreign_surface)
{
    GStreamerGWorld *gstGWorld = static_cast<GStreamerGWorld*>(data);

    gstGWorld->expose();
}

static const struct fbx_foreign_surface_listener foreign_surface_listener = {
    foreign_surface_id,
    foreign_surface_imported,
};

GStreamerGWorld::GStreamerGWorld(GstElement* pipeline)
    : m_pipeline(pipeline)
    , wl_compositor(0)
    , wl_surface(0)
    , fbx_fsm(0)
    , fbx_foreign_surface(0)
    , m_overlay_id(0)
{
    // Overlay messages need to be handled synchronously.
    GRefPtr<GstBus> bus = webkitGstPipelineGetBus(GST_PIPELINE(m_pipeline));
    gst_bus_set_sync_handler(bus.get(), gstGWorldSyncMessageCallback, this, 0);

    QPlatformNativeInterface *native = QGuiApplication::platformNativeInterface();
    struct wl_display *display = (struct wl_display *)
        native->nativeResourceForIntegration("wl_display");

    struct wl_event_queue *queue = (struct wl_event_queue *)
        native->nativeResourceForIntegration("wl_event_queue");

    if (!display)
        return;

    gst_element_set_context(pipeline,
            gst_wayland_display_handle_context_new(display));

    struct wl_registry *registry = wl_display_get_registry(display);
    wl_registry_add_listener(registry, &wl_registry_listener, this);
    if (queue) {
        wl_proxy_set_queue((struct wl_proxy *)registry, queue);
        wl_display_roundtrip_queue(display, queue);
    } else {
        wl_display_roundtrip(display);
    }

    if (wl_compositor && fbx_fsm) {
        wl_surface = wl_compositor_create_surface(wl_compositor);
        fbx_foreign_surface = fbx_foreign_surface_manager_export_surface(fbx_fsm, wl_surface);
        fbx_foreign_surface_add_listener(fbx_foreign_surface, &foreign_surface_listener, this);

        if (queue)
            wl_display_roundtrip_queue(display, queue);
        else
            wl_display_roundtrip(display);
    }
}

GStreamerGWorld::~GStreamerGWorld()
{
    exitFullscreen();

    GRefPtr<GstBus> bus = webkitGstPipelineGetBus(GST_PIPELINE(m_pipeline));
    gst_bus_set_sync_handler(bus.get(), 0, this, 0);

    if (wl_surface)
        wl_surface_destroy(wl_surface);

    if (wl_compositor)
        wl_compositor_destroy(wl_compositor);

    if (fbx_foreign_surface)
        fbx_foreign_surface_destroy(fbx_foreign_surface);

    if (fbx_fsm)
        fbx_foreign_surface_manager_destroy(fbx_fsm);

    m_pipeline = 0;
}

bool GStreamerGWorld::hasOverlay() const
{
    return m_overlay_id != 0;
}

uint32_t GStreamerGWorld::overlayID() const
{
    return m_overlay_id;
}

bool GStreamerGWorld::enterFullscreen()
{
    return false;
}

void GStreamerGWorld::exitFullscreen()
{
}

void GStreamerGWorld::setWindowOverlay(GstMessage* message)
{
    GstObject* sink = GST_MESSAGE_SRC(message);

    if (!GST_IS_VIDEO_OVERLAY(sink))
        return;

    if (g_object_class_find_property(G_OBJECT_GET_CLASS(sink), "force-aspect-ratio"))
        g_object_set(sink, "force-aspect-ratio", TRUE, NULL);

    if (wl_surface) {
        gst_video_overlay_set_window_handle(GST_VIDEO_OVERLAY(sink),
            (guintptr)wl_surface);
    }
}

void GStreamerGWorld::expose()
{
    if (!GST_IS_VIDEO_OVERLAY(m_pipeline))
        return;

    gst_video_overlay_expose(GST_VIDEO_OVERLAY(m_pipeline));
}

}
#endif // ENABLE(VIDEO) && USE(GSTREAMER)
