/*
 * Copyright (C) 2012 Nokia Corporation and/or its subsidiary(-ies)
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebPageProxy.h"

#include "PageClient.h"
#include "UserAgentQt.h"
#include "WebKitVersion.h"
#include "WebPageMessages.h"
#include "WebProcessProxy.h"
#include <WebCore/Editor.h>
#include <WebCore/NotImplemented.h>

#include <QtGui/QGuiApplication>
#include <QtGui/qpa/qplatformnativeinterface.h>

#if HAVE(QTQUICK)
#include "QtNetworkReplyData.h"
#include "QtPageClient.h"
#include "qquicknetworkreply_p.h"
#endif

#include <wayland-client.h>
#include <wayland/freebox.h>

using namespace WebCore;

namespace WebKit {

String WebPageProxy::standardUserAgent(const String& applicationNameForUserAgent)
{
    return UserAgentQt::standardUserAgent(applicationNameForUserAgent, WEBKIT_MAJOR_VERSION, WEBKIT_MINOR_VERSION);
}

void WebPageProxy::saveRecentSearches(const String&, const Vector<String>&)
{
    notImplemented();
}

void WebPageProxy::loadRecentSearches(const String&, Vector<String>&)
{
    notImplemented();
}

void WebPageProxy::registerApplicationScheme(const String& scheme)
{
    process()->send(Messages::WebPage::RegisterApplicationScheme(scheme), m_pageID);
}

void WebPageProxy::resolveApplicationSchemeRequest(QtNetworkRequestData request)
{
#if HAVE(QTQUICK)
    RefPtr<QtRefCountedNetworkRequestData> requestData = adoptRef(new QtRefCountedNetworkRequestData(request));
    m_applicationSchemeRequests.add(requestData);
    static_cast<QtPageClient*>(m_pageClient)->handleApplicationSchemeRequest(requestData);
#endif
}

void WebPageProxy::sendApplicationSchemeReply(const QQuickNetworkReply* reply)
{
#if HAVE(QTQUICK)
    RefPtr<QtRefCountedNetworkRequestData> requestData = reply->networkRequestData();
    if (m_applicationSchemeRequests.contains(requestData)) {
        RefPtr<QtRefCountedNetworkReplyData> replyData = reply->networkReplyData();
        process()->send(Messages::WebPage::ApplicationSchemeReply(replyData->data()), pageID());
        m_applicationSchemeRequests.remove(requestData);
    }
#endif
}

void WebPageProxy::authenticationRequiredRequest(const String& hostname, const String& realm, const String& prefilledUsername, String& username, String& password)
{
    m_pageClient->handleAuthenticationRequiredRequest(hostname, realm, prefilledUsername, username, password);
}

void WebPageProxy::proxyAuthenticationRequiredRequest(const String& hostname, uint16_t port, const String& prefilledUsername, String& username, String& password)
{
    m_pageClient->handleProxyAuthenticationRequiredRequest(hostname, port, prefilledUsername, username, password);
}

void WebPageProxy::certificateVerificationRequest(const String& hostname, bool& ignoreErrors)
{
    m_pageClient->handleCertificateVerificationRequest(hostname, ignoreErrors);
}

#if PLUGIN_ARCHITECTURE(X11)
void WebPageProxy::createPluginContainer(uint64_t& windowID)
{
    notImplemented();
}

void WebPageProxy::windowedPluginGeometryDidChange(const WebCore::IntRect& frameRect, const WebCore::IntRect& clipRect, uint64_t windowID)
{
    notImplemented();
}
#endif

static struct fbx_foreign_surface_manager *fbx_fsm;
static struct wl_subcompositor *wl_subcompositor;
static bool wl_initialized;

void
registry_handle_global(void *data, struct wl_registry *registry,
               uint32_t id, const char *interface, uint32_t version)
{
    if (!strcmp(interface, "wl_subcompositor")) {
        wl_subcompositor = (struct wl_subcompositor *)
            wl_registry_bind(registry, id,
                    &wl_subcompositor_interface, 1);
    } else if (!strcmp(interface, "fbx_foreign_surface_manager")) {
        fbx_fsm = (struct fbx_foreign_surface_manager *)
            wl_registry_bind(registry, id,
                    &fbx_foreign_surface_manager_interface, 1);
    }
}

static void
registry_handle_global_remove(void *data, struct wl_registry *registry,
                  uint32_t name)
{
}

static const struct wl_registry_listener wl_registry_listener = {
    registry_handle_global,
    registry_handle_global_remove,
};

void WebPageProxy::attachOverlay(uint32_t surfaceID, const WebCore::IntRect& rect)
{
    struct wl_subsurface *subsurface = static_cast<struct wl_subsurface *>(m_overlays.get(surfaceID));

    if (!subsurface) {
            QPlatformNativeInterface *native = QGuiApplication::platformNativeInterface();

        if (!wl_initialized) {
            struct wl_display *display = (struct wl_display *)
                native->nativeResourceForIntegration("wl_display");

            fbx_fsm = 0;

            if (display) {
                struct wl_registry *registry = wl_display_get_registry(display);
                wl_registry_add_listener(registry, &wl_registry_listener, this);
                wl_display_roundtrip(display);
            }

            wl_initialized = true;
        }

        if (fbx_fsm && wl_subcompositor) {
            struct wl_surface *surface = (struct wl_surface *)
                native->nativeResourceForWindow("surface", QGuiApplication::topLevelAt(QPoint()));

            if (surface) {
                struct ::wl_surface *foreign_surface =
                    fbx_foreign_surface_manager_import_surface(fbx_fsm, surfaceID);

                subsurface = wl_subcompositor_get_subsurface(wl_subcompositor,
                        foreign_surface, surface);

                wl_subsurface_place_below(subsurface, surface);
                wl_subsurface_set_desync(subsurface);

                m_overlays.add(surfaceID, subsurface);
            }
        }
    }

    if (subsurface)
        wl_subsurface_set_position(subsurface, rect.x(), rect.y());
}

void WebPageProxy::detachOverlay(uint32_t surfaceID)
{
    struct wl_subsurface *subsurface = static_cast<struct wl_subsurface *>(m_overlays.take(surfaceID));

    if (subsurface)
        wl_subsurface_destroy(subsurface);
}

void WebPageProxy::changeSelectedIndex(int32_t selectedIndex)
{
    process()->send(Messages::WebPage::SelectedIndex(selectedIndex), m_pageID);
}

void WebPageProxy::closePopupMenu()
{
    process()->send(Messages::WebPage::HidePopupMenu(), m_pageID);
}

void WebPageProxy::willSetInputMethodState()
{
    m_pageClient->handleWillSetInputMethodState();
}

} // namespace WebKit
