/****************************************************************************
**
** Copyright (C) 2013 Klarälvdalens Datakonsult AB (KDAB).
** Contact: http://www.qt.io/licensing/
**
** This file is part of the Qt Compositor.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of The Qt Company Ltd nor the names of its
**     contributors may be used to endorse or promote products derived
**     from this software without specific prior written permission.
**
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qwaylandinputcontext_p.h"

#include <QGuiApplication>
#include <QTextFormat>
#include <QPalette>
#include <QWindow>
#ifndef QT_NO_WAYLAND_XKB
#include <xkbcommon/xkbcommon.h>
#endif

#include "qwaylanddisplay_p.h"
#include "qwaylandinputdevice_p.h"
#include "qwaylandwindow_p.h"

QT_BEGIN_NAMESPACE

namespace QtWaylandClient {

static Qt::Key toQtKey(uint32_t sym)
{
#ifndef QT_NO_WAYLAND_XKB
    switch (static_cast<xkb_keysym_t>(sym)) {
    case XKB_KEY_BackSpace:
        return Qt::Key_Backspace;
    case XKB_KEY_Return:
        return Qt::Key_Return;
    case XKB_KEY_Left:
        return Qt::Key_Left;
    case XKB_KEY_Up:
        return Qt::Key_Up;
    case XKB_KEY_Right:
        return Qt::Key_Right;
    case XKB_KEY_Down:
        return Qt::Key_Down;
    default:
        return Qt::Key_unknown;
    }
#else
    return Qt::Key_unknown;
#endif
}

static QEvent::Type toQEventType(uint32_t state)
{
    switch (static_cast<wl_keyboard_key_state>(state)) {
    default:
    case WL_KEYBOARD_KEY_STATE_PRESSED:
        return QEvent::KeyPress;
    case WL_KEYBOARD_KEY_STATE_RELEASED:
        return QEvent::KeyRelease;
    }
}

QWaylandTextInput::QWaylandTextInput(struct ::wl_text_input *text_input)
    : QtWayland::wl_text_input(text_input)
    , m_commit()
    , m_commitIndex(0)
    , m_commitLength(0)
    , m_serial(0)
    , m_resetSerial(0)
    , m_enabled(false)
    , m_shiftMask(0)
    , m_ctrlMask(0)
    , m_altMask(0)
    , m_metaMask(0)
{
}

QString QWaylandTextInput::commitString() const
{
    return m_commit;
}

void QWaylandTextInput::reset()
{
    m_commit.clear();
    m_commitIndex = 0;
    m_commitLength = 0;
    m_preeditAttributes.clear();
    wl_text_input::reset();
    m_resetSerial = m_serial;
}

bool QWaylandTextInput::enabled() const
{
    return m_enabled;
}

bool QWaylandTextInput::query(Qt::InputMethodQueries queries)
{
    if (!QGuiApplication::focusObject())
        return false;

    // we only care about a few properties
    const Qt::InputMethodQueries surroundingTextQueries =
        Qt::ImSurroundingText | Qt::ImCursorPosition | Qt::ImAnchorPosition;

    Qt::InputMethodQueries req = Qt::ImEnabled | Qt::ImHints;
    if (queries & surroundingTextQueries)
        req |= surroundingTextQueries;

    QInputMethodQueryEvent event(req);
    QCoreApplication::sendEvent(QGuiApplication::focusObject(), &event);

    bool enabled = event.value(Qt::ImEnabled).toBool();
    Qt::InputMethodHints hints = static_cast<Qt::InputMethodHints>(event.value(Qt::ImHints).toUInt());

    // hack for freebox
    enabled &= !(hints & Qt::ImhDigitsOnly);
    if (enabled) {
        bool commit = false;

        if (queries & Qt::ImHints) {
            uint32_t content_purpose;
            uint32_t content_hints = 0;

            if (hints & Qt::ImhHiddenText)
                content_hints |= wl_text_input::content_hint_hidden_text;
            if (hints & Qt::ImhSensitiveData)
                content_hints |= wl_text_input::content_hint_sensitive_data;
            if (!(hints & (Qt::ImhNoAutoUppercase | Qt::ImhLowercaseOnly)))
                content_hints |= wl_text_input::content_hint_auto_capitalization;
            if (hints & (Qt::ImhPreferUppercase | Qt::ImhUppercaseOnly))
                content_hints |= wl_text_input::content_hint_uppercase;
            if (hints & (Qt::ImhPreferLowercase | Qt::ImhLowercaseOnly))
                content_hints |= wl_text_input::content_hint_lowercase;
            if (!(hints & Qt::ImhNoPredictiveText))
                content_hints |= wl_text_input::content_hint_auto_completion |
                    wl_text_input::content_hint_auto_correction;
            if (hints & (Qt::ImhPreferLatin | Qt::ImhLatinOnly))
                content_hints |= wl_text_input::content_hint_latin;
            if (hints & Qt::ImhMultiLine)
                content_hints |= wl_text_input::content_hint_multiline;

            if (hints & (Qt::ImhDate | Qt::ImhTime))
                content_purpose = wl_text_input::content_purpose_datetime;
            else if (hints & Qt::ImhDate)
                content_purpose = wl_text_input::content_purpose_date;
            else if (hints & Qt::ImhTime)
                content_purpose = wl_text_input::content_purpose_time;
            else if (hints & Qt::ImhDigitsOnly)
                content_purpose = wl_text_input::content_purpose_digits;
            else if (hints & Qt::ImhFormattedNumbersOnly)
                content_purpose = wl_text_input::content_purpose_number;
            else if (hints & (Qt::ImhUppercaseOnly | Qt::ImhLowercaseOnly | Qt::ImhLatinOnly))
                content_purpose = wl_text_input::content_purpose_alpha;
            else if (hints & Qt::ImhDialableCharactersOnly)
                content_purpose = wl_text_input::content_purpose_phone;
            else if (hints & Qt::ImhEmailCharactersOnly)
                content_purpose = wl_text_input::content_purpose_email;
            else if (hints & Qt::ImhUrlCharactersOnly)
                content_purpose = wl_text_input::content_purpose_url;
            else if ((hints & Qt::ImhHiddenText) && (hints & Qt::ImhSensitiveData))
                content_purpose = wl_text_input::content_purpose_password;
            else
                content_purpose = wl_text_input::content_purpose_normal;

            if (content_purpose != wl_text_input::content_purpose_normal &&
                content_purpose != wl_text_input::content_purpose_alpha)
                content_hints &= ~wl_text_input::content_hint_auto_capitalization;

            set_content_type(content_hints, content_purpose);
            commit = true;
        }

        if (queries & Qt::ImSurroundingText) {
            const QString &text = event.value(Qt::ImSurroundingText).toString();
            const int cursor = event.value(Qt::ImCursorPosition).toInt();
            const int anchor = event.value(Qt::ImAnchorPosition).toInt();

            set_surrounding_text(text, cursor, anchor);
            commit = true;
        }

        if (commit)
            commit_state(++m_serial);
    }

    return enabled;
}

void QWaylandTextInput::updateState(Qt::InputMethodQueries queries)
{
    bool enable = query(queries);
    if (m_enabled != enable) {
        m_enabled = enable;
        emit enabledChanged(m_enabled);
    }
}

void QWaylandTextInput::text_input_preedit_cursor(int32_t index)
{
    m_preeditAttributes <<
        QInputMethodEvent::Attribute(QInputMethodEvent::Cursor,
                index, index >= 0, QVariant());
}

void QWaylandTextInput::text_input_preedit_styling(uint32_t index, uint32_t length, uint32_t style)
{
    QTextCharFormat format;
    QPalette palette;

    switch (style) {
    default:
    case wl_text_input::preedit_style_none:
        break;
    case wl_text_input::preedit_style_active:
    case wl_text_input::preedit_style_highlight:
        format.setForeground(palette.color(QPalette::Active, QPalette::Base));
        break;
    case wl_text_input::preedit_style_inactive:
        format.setForeground(palette.color(QPalette::Inactive, QPalette::Base));
        break;
    case wl_text_input::preedit_style_default:
    case wl_text_input::preedit_style_underline:
        format.setFontUnderline(true);
        break;
    case wl_text_input::preedit_style_selection:
        format.setBackground(palette.color(QPalette::Active, QPalette::Highlight));
        format.setForeground(palette.color(QPalette::Active, QPalette::HighlightedText));
        break;
    case wl_text_input::preedit_style_incorrect:
        format.setFontUnderline(true);
        format.setUnderlineStyle(QTextCharFormat::SpellCheckUnderline);
        format.setUnderlineColor(Qt::red);
        break;
    }

    m_preeditAttributes <<
        QInputMethodEvent::Attribute(QInputMethodEvent::TextFormat,
                index, length, format);
}

void QWaylandTextInput::text_input_preedit_string(uint32_t serial, const QString &text, const QString &commit)
{
    if (serial < m_resetSerial) {
        qWarning("ignore preedit: serial %u, current %u, reset %u",
                serial, m_serial, m_resetSerial);
        return;
    }

    if (!QGuiApplication::focusObject())
        return;

    QInputMethodEvent event(text, m_preeditAttributes);
    m_preeditAttributes.clear();
    QCoreApplication::sendEvent(QGuiApplication::focusObject(), &event);

    m_commit = commit;
}

void QWaylandTextInput::text_input_commit_string(uint32_t serial, const QString &text)
{
    if (serial < m_resetSerial) {
        qWarning("ignore commit: serial %u, current %u, reset %u",
                serial, m_serial, m_resetSerial);
        return;
    }

    if (!QGuiApplication::focusObject())
        return;

    QInputMethodEvent event;
    event.setCommitString(text, m_commitIndex, m_commitLength);
    QCoreApplication::sendEvent(QGuiApplication::focusObject(), &event);

    m_commit.clear();
    m_commitIndex = 0;
    m_commitLength = 0;
    m_preeditAttributes.clear();
}

void QWaylandTextInput::text_input_enter(wl_surface *)
{
    updateState();
    m_resetSerial = m_serial;
}

void QWaylandTextInput::text_input_leave()
{
    if (!m_commit.isEmpty())
        text_input_commit_string(0, m_commit);
}

void QWaylandTextInput::text_input_delete_surrounding_text(int32_t index, uint32_t length)
{
    m_commitIndex = index;
    m_commitLength = length;
}

void QWaylandTextInput::text_input_keysym(uint32_t serial, uint32_t time, uint32_t sym, uint32_t state, uint32_t modifiers)
{
    Q_UNUSED(serial);
    Q_UNUSED(time);

    if (!QGuiApplication::focusWindow())
        return;

    Qt::KeyboardModifiers qtModifiers = Qt::NoModifier;
    if (modifiers & m_shiftMask)
        qtModifiers |= Qt::ShiftModifier;
    if (modifiers & m_ctrlMask)
        qtModifiers |= Qt::ControlModifier;
    if (modifiers & m_altMask)
        qtModifiers |= Qt::AltModifier;
    if (modifiers & m_metaMask)
        qtModifiers |= Qt::MetaModifier;

    QEvent::Type type = toQEventType(state);

    Qt::Key qtKey = toQtKey(sym);
    if (type == QEvent::KeyPress &&
            (qtKey == Qt::Key_Return || qtKey == Qt::Key_Enter ||
             qtKey == Qt::Key_Escape || qtKey == Qt::Key_Back))
        emit done();

    QWindowSystemInterface::handleKeyEvent(QGuiApplication::focusWindow(),
            type, qtKey, qtModifiers);
}

void QWaylandTextInput::text_input_modifiers_map(wl_array *map)
{
    const char *data = static_cast<const char *>(map->data);
    uint32_t index = 0;

    m_shiftMask = 0;
    m_ctrlMask = 0;
    m_altMask = 0;
    m_metaMask = 0;

    for (const char *p = data; p < data + map->size; p += qstrlen(p) + 1) {
        if (!qstrcmp(p, "Shift"))
            m_shiftMask = 1 << index;
        else if (!qstrcmp(p, "Control"))
            m_ctrlMask = 1 << index;
        else if (!qstrcmp(p, "Mod1"))
            m_altMask = 1 << index;
        else if (!qstrcmp(p, "Mod4"))
            m_metaMask = 1 << index;

        index++;
    }
}

void QWaylandTextInput::text_input_input_panel_state(uint32_t state)
{
    emit inputPanelVisibilityChanged(!!state);
}

QWaylandInputContext::QWaylandInputContext(QWaylandDisplay *display)
    : QPlatformInputContext()
    , mDisplay(display)
    , mTextInput()
    , mShow(false)
    , mVisible(false)
    , mActivated(false)
    , mDone(false)
{
}

bool QWaylandInputContext::isValid() const
{
    return mDisplay->textInputManager() != 0;
}

bool QWaylandInputContext::hasCapability(Capability capability) const
{
    switch (capability) {
    case HiddenTextCapability:
        return true;
    }

    return false;
}

void QWaylandInputContext::reset()
{
    mTextInput->reset();
}

void QWaylandInputContext::commit()
{
    if (!mActivated)
        return;

    if (!ensureTextInput())
        return;

    if (!QGuiApplication::focusObject())
        return;

    QInputMethodEvent event;
    event.setCommitString(mTextInput->commitString());
    QCoreApplication::sendEvent(QGuiApplication::focusObject(), &event);

    mTextInput->reset();
}

void QWaylandInputContext::update(Qt::InputMethodQueries queries)
{
    if (!ensureTextInput())
        return;

    mTextInput->updateState(queries);
}

void QWaylandInputContext::invokeAction(QInputMethod::Action action, int cursorPosition)
{
    int button = 0;

    if (!ensureTextInput())
        return;

    switch (action) {
    case QInputMethod::Click:
        button = 0x110; // kernel BTN_LEFT
        break;
    case QInputMethod::ContextMenu:
        button = 0x111; // kernel BTN_RIGHT
        break;
    }

    if (button) {
        mTextInput->invoke_action(button, cursorPosition); // FIXME to UTF8 cursor position
    }
}

bool QWaylandInputContext::filterEvent(const QEvent *event)
{
    if (!mTextInput)
        return false;

    switch (event->type()) {
    case QEvent::KeyPress: {
        const QKeyEvent *keyEvent = static_cast<const QKeyEvent *>(event);
        commit();
        if (!keyEvent->isAutoRepeat() &&
                (keyEvent->key() == Qt::Key_Return ||
                 keyEvent->key() == Qt::Key_Enter ||
                 keyEvent->key() == Qt::Key_Escape ||
                 keyEvent->key() == Qt::Key_Back))
            textInputDone();
        break;
    }
    default:
        break;
    }

    return false;
}

void QWaylandInputContext::checkActivate()
{
    if (inputMethodAccepted() && mShow && mTextInput->enabled() && !mDone)
        activateTextInput();
    else
        deactivateTextInput();
}

void QWaylandInputContext::showInputPanel()
{
    mShow = true;

    if (!ensureTextInput())
        return;

    checkActivate();

    mTextInput->show_input_panel();
}

void QWaylandInputContext::hideInputPanel()
{
    mShow = false;

    if (!ensureTextInput())
        return;

    checkActivate();

    mTextInput->hide_input_panel();
}

bool QWaylandInputContext::isInputPanelVisible() const
{
    return mVisible;
}

void QWaylandInputContext::inputPanelVisibilityChanged(bool visible)
{
    if (mVisible != visible) {
        mVisible = visible;
        // do not emit visibility changed when the keyboard is hidden on
        // commit, otherwise the keyboard will be shown again right away
        if (mVisible || !mDone)
            emitInputPanelVisibleChanged();
    }
}

void QWaylandInputContext::textInputEnabledChanged(bool enabled)
{
    if (enabled)
        mDone = false;
    checkActivate();
}

void QWaylandInputContext::textInputDone()
{
    mDone = true;
    checkActivate();
}

void QWaylandInputContext::setFocusObject(QObject *object)
{
    if (!ensureTextInput())
        return;

    mDone = false;
    mTextInput->updateState(mActivated ? Qt::ImQueryAll : Qt::ImEnabled);
    checkActivate();
}

void QWaylandInputContext::deactivateTextInput()
{
    if (!mActivated)
        return;

    mTextInput->deactivate(mDisplay->defaultInputDevice()->wl_seat());
    mTextInput->reset();
    mActivated = false;
}

void QWaylandInputContext::activateTextInput()
{
    if (mActivated)
        return;

    QWindow *window = QGuiApplication::focusWindow();
    if (!window || !window->handle())
        return;

    struct ::wl_surface *surface = static_cast<QWaylandWindow *>(window->handle())->object();
    mTextInput->activate(mDisplay->defaultInputDevice()->wl_seat(), surface);

    mActivated = true;
}

bool QWaylandInputContext::ensureTextInput()
{
    if (mTextInput)
        return true;

    if (!isValid())
        return false;

    mTextInput.reset(new QWaylandTextInput(mDisplay->textInputManager()->create_text_input()));
    connect(mTextInput.data(), &QWaylandTextInput::inputPanelVisibilityChanged,
            this, &QWaylandInputContext::inputPanelVisibilityChanged);
    connect(mTextInput.data(), &QWaylandTextInput::enabledChanged,
            this, &QWaylandInputContext::textInputEnabledChanged);
    connect(mTextInput.data(), &QWaylandTextInput::done,
            this, &QWaylandInputContext::textInputDone);

    return true;
}

}

QT_END_NAMESPACE

