#include "qsgholenode.h"

#include <QtGui/qopenglfunctions.h>

QT_BEGIN_NAMESPACE

class HoleMaterialShader : public QSGMaterialShader
{
public:
    void initialize();
    void activate();
    void deactivate();
    void updateState(const RenderState &state, QSGMaterial *newEffect, QSGMaterial *oldEffect);

    char const *const *attributeNames() const;

    static QSGMaterialType type;

private:
    const char *vertexShader() const;
    const char *fragmentShader() const;

    int m_matrix_id;
    float m_alpha;
};

QSGMaterialType HoleMaterialShader::type;

void HoleMaterialShader::activate()
{
    QOpenGLFunctions *funcs = QOpenGLContext::currentContext()->functions();
    funcs->glBlendColor(0, 0, 0, 1.0f - m_alpha);
    funcs->glBlendFunc(GL_ZERO, GL_CONSTANT_ALPHA);
}

void HoleMaterialShader::deactivate()
{
    QOpenGLFunctions *funcs = QOpenGLContext::currentContext()->functions();
    funcs->glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
}

void HoleMaterialShader::updateState(const RenderState &state, QSGMaterial *newEffect, QSGMaterial *oldEffect)
{
    m_alpha = state.opacity();

    if (state.isMatrixDirty())
        program()->setUniformValue(m_matrix_id, state.combinedMatrix());
}

char const *const *HoleMaterialShader::attributeNames() const
{
    static char const *const attr[] = { "position", 0 };
    return attr;
}

void HoleMaterialShader::initialize()
{
    m_matrix_id = program()->uniformLocation("matrix");
}

const char *HoleMaterialShader::vertexShader() const
{
    return "\n uniform highp mat4 matrix;"
           "\n attribute highp vec4 position;"
           "\n void main(void)"
           "\n {"
           "\n     gl_Position = matrix * position;"
           "\n }";
}

const char *HoleMaterialShader::fragmentShader() const
{
    return "\n void main(void)"
           "\n {"
           "\n     gl_FragColor = vec4(0, 0, 0, 0);"
           "\n }";
}

QSGHoleMaterial::QSGHoleMaterial()
{
    setFlag(Blending, false);
}

QSGMaterialType *QSGHoleMaterial::type() const
{
    return &HoleMaterialShader::type;
}

QSGMaterialShader *QSGHoleMaterial::createShader() const
{
    return new HoleMaterialShader;
}

QSGHoleNode::QSGHoleNode(const QRectF &rect)
    : m_geometry(QSGGeometry::defaultAttributes_Point2D(), 4)
{
    QSGGeometry::updateRectGeometry(&m_geometry, rect);
    setMaterial(&m_material);
    setGeometry(&m_geometry);
}

QSGHoleNode::QSGHoleNode()
    : m_geometry(QSGGeometry::defaultAttributes_Point2D(), 4)
{
    QSGGeometry::updateRectGeometry(&m_geometry, QRectF());
    setMaterial(&m_material);
    setGeometry(&m_geometry);
}

void QSGHoleNode::setRect(const QRectF &rect)
{
    QSGGeometry::updateRectGeometry(&m_geometry, rect);
    markDirty(QSGNode::DirtyGeometry);
}

QRectF QSGHoleNode::rect() const
{
    const QSGGeometry::Point2D *pts = m_geometry.vertexDataAsPoint2D();
    return QRectF(pts[0].x,
                  pts[0].y,
                  pts[3].x - pts[0].x,
                  pts[3].y - pts[0].y);
}

QT_END_NAMESPACE
