#include "qfbxdeclarativevideooutput.h"
#include "foreign_surface_manager.h"
#include "qdeclarativevideooutput_p.h"
#include "qsgholenode.h"

#include <QtGui/qguiapplication.h>
#include <QtGui/qscreen.h>
#include <QtGui/qpa/qplatformnativeinterface.h>
#include <QtGui/qpa/qplatformscreen.h>
#include <QtQuick/qquickwindow.h>
#include <QtMultimedia/qmediaservice.h>
#include <QtMultimedia/qvideowindowcontrol.h>

QT_BEGIN_NAMESPACE

class QFbxHoleNode : public QSGHoleNode
{
public:
    QFbxHoleNode()
        : QSGHoleNode()
        , preprocessed(false)
    {
        setFlag(QSGNode::UsePreprocess, true);
    }

    void preprocess()
    {
        preprocessed = true;
    }

    bool preprocessed;
};

QFbxDeclarativeVideoOutput::QFbxDeclarativeVideoOutput(QDeclarativeVideoOutput *parent)
    : QDeclarativeVideoBackend(parent)
    , m_videoWindowControl(0)
    , m_manager(0)
    , m_window(0)
    , m_node(0)
    , m_resizing(false)
    , m_dirtySize(false)
    , m_dirtyPosition(false)
{
}

QFbxDeclarativeVideoOutput::~QFbxDeclarativeVideoOutput()
{
    unbindWindow();
    releaseSource();
    releaseControl();
}

bool QFbxDeclarativeVideoOutput::init(QMediaService *service)
{
    QMediaControl *control = service->requestControl(QVideoWindowControl_iid);
    if (!control)
        return false;

    m_videoWindowControl = qobject_cast<QVideoWindowControl *>(control);
    if (!m_videoWindowControl)
        return false;

    m_service = service;
    QObject::connect(m_videoWindowControl.data(),
                     SIGNAL(nativeSizeChanged()),
                     q, SLOT(_q_updateNativeSize()));

    m_manager = fbx::ForeignSurfaceManager::getInstance();
    connect(m_manager, &fbx::ForeignSurfaceManager::connected,
            this, &QFbxDeclarativeVideoOutput::bindWindow);

    bindWindow();

    q->setFlag(QQuickItem::ItemHasContents, true);
    m_dirtySize = true;

    return true;
}

void QFbxDeclarativeVideoOutput::itemChange(QQuickItem::ItemChange change,
        const QQuickItem::ItemChangeData &changeData)
{
    switch (change) {
    case QQuickItem::ItemVisibleHasChanged:
    case QQuickItem::ItemSceneChange:
        rebindWindow();
        break;
    case QQuickItem::ItemDevicePixelRatioHasChanged:
        q->polish();
        break;
    default:
        break;
    }
}

void QFbxDeclarativeVideoOutput::releaseSource()
{
}

void QFbxDeclarativeVideoOutput::releaseControl()
{
    if (m_videoWindowControl) {
        if (m_service)
            m_service->releaseControl(m_videoWindowControl);
        m_videoWindowControl = 0;
    }
}

QSize QFbxDeclarativeVideoOutput::nativeSize() const
{
    return m_videoWindowControl->nativeSize();
}

void QFbxDeclarativeVideoOutput::updateGeometry()
{
    switch (q->fillMode()) {
    case QDeclarativeVideoOutput::PreserveAspectFit:
        m_videoWindowControl->setAspectRatioMode(Qt::KeepAspectRatio);
        break;
    case QDeclarativeVideoOutput::PreserveAspectCrop:
        m_videoWindowControl->setAspectRatioMode(Qt::KeepAspectRatioByExpanding);
        break;
    case QDeclarativeVideoOutput::Stretch:
        m_videoWindowControl->setAspectRatioMode(Qt::IgnoreAspectRatio);
        break;
    };
}

QSGNode *QFbxDeclarativeVideoOutput::updatePaintNode(QSGNode *node,
        QQuickItem::UpdatePaintNodeData *data)
{
    Q_UNUSED(data);

    QFbxHoleNode *hole = static_cast<QFbxHoleNode *>(node);
    if (!hole)
        hole = new QFbxHoleNode;

    hole->setRect(q->boundingRect());

    m_node = hole;

    return hole;
}

QAbstractVideoSurface *QFbxDeclarativeVideoOutput::videoSurface() const
{
    return 0;
}

QRectF QFbxDeclarativeVideoOutput::adjustedViewport() const
{
    return QRectF(QPointF(0, 0), nativeSize());
}

void QFbxDeclarativeVideoOutput::unbindWindow()
{
    if (m_window) {
        unbindWaylandSurface();

        disconnect(m_window, &QQuickWindow::beforeSynchronizing,
                   this, &QFbxDeclarativeVideoOutput::syncItem);
        disconnect(m_window, &QQuickWindow::afterRendering,
                   this, &QFbxDeclarativeVideoOutput::syncNode);
        disconnect(m_window, &QQuickWindow::frameSwapped,
                   this, &QFbxDeclarativeVideoOutput::frameSwapped);
        m_window = 0;
    }
}

void QFbxDeclarativeVideoOutput::rebindWindow()
{
    unbindWindow();
    bindWindow();
}

void QFbxDeclarativeVideoOutput::bindWindow()
{
    if (m_window)
        return;

    if (!q->isVisible())
        return;

    if (!m_manager->isConnected())
        return;

    QQuickWindow *w = q->window();
    if (w)
        w->create();

    m_window = w;
    if (m_window) {
        connect(m_window, &QQuickWindow::beforeSynchronizing,
                this, &QFbxDeclarativeVideoOutput::syncItem, Qt::DirectConnection);
        connect(m_window, &QQuickWindow::afterRendering,
                this, &QFbxDeclarativeVideoOutput::syncNode, Qt::DirectConnection);
        connect(m_window, &QQuickWindow::frameSwapped,
                this, &QFbxDeclarativeVideoOutput::frameSwapped, Qt::DirectConnection);
    }
}

bool QFbxDeclarativeVideoOutput::bindWaylandSurface()
{
    Q_ASSERT(m_window && m_manager->isConnected());

    if (wl_subsurface::isInitialized())
        return true;

    WId id = m_videoWindowControl->winId();
    if (!id)
        return false;

    QPlatformNativeInterface *native = QGuiApplication::platformNativeInterface();
    struct ::wl_surface *parent = static_cast<struct wl_surface *>(
            native->nativeResourceForWindow("surface", m_window));

    struct ::wl_surface *foreign_surface = m_manager->import_surface(id);
    wl_subsurface::init(m_manager->get_subsurface(foreign_surface, parent));
    wl_subsurface::place_below(parent);
    wl_surface_destroy(foreign_surface);

    m_dirtyPosition = true;

    return true;
}

void QFbxDeclarativeVideoOutput::unbindWaylandSurface()
{
    m_dirtyPosition = false;
    m_resizing = false;

    if (wl_subsurface::isInitialized())
        wl_subsurface::destroy();
}

QRectF QFbxDeclarativeVideoOutput::absoluteGeometry() const
{
    QQuickWindow *win = q->window();
    qreal devicePixelRatio;

    if (!win)
        devicePixelRatio = qApp->devicePixelRatio();
    else {
        devicePixelRatio = win->effectiveDevicePixelRatio();
        if (win->screen())
            devicePixelRatio /= win->screen()->handle()->devicePixelRatio();
    }

    QRectF rect = q->mapRectToScene(QRectF(0, 0, q->width(), q->height()));

    return QRectF(rect.topLeft() * devicePixelRatio,
                  rect.size() * devicePixelRatio);
}

/*
 * Record the QQuickItem attributes that will need to be applied on the
 * wayland video subsurface.
 *
 * This is called before rendering the scenegraph, while the GUI thread is
 * locked
 */
void QFbxDeclarativeVideoOutput::syncItem()
{
    QRect rect = absoluteGeometry().toRect();

    if (rect.topLeft() != m_absolutePos)
        m_dirtyPosition = true;

    m_absolutePos = rect.topLeft();

    if (rect.size() != m_size)
        m_dirtySize = true;

    m_size = rect.size();

    // set size now, the fbxbus call must be done while the GUI thread is locked
    if (m_dirtySize) {
        beginGeometryChange();
        m_videoWindowControl->setDisplayRect(QRect(QPoint(0, 0), m_size));
        m_dirtySize = false;
    }
}

/*
 * Synchronize the video output state with the wayland video subsurface.
 *
 * This function is called after the window has been rendered from the
 * scenegraph thread.  At this point are able to determine if the video
 * subsurface should be displayed or not: the scenegraph node will have its
 * @preprocessed flag set only if it is effectively visible for this frame. If
 * it's not set, it means either its inherited opacity is 0 or the QQuickItem
 * subtree is not visible.
 */
void QFbxDeclarativeVideoOutput::syncNode()
{
    bool visible = m_node && m_node->preprocessed;

    if (!visible) {
        unbindWaylandSurface();
    } else if (bindWaylandSurface()) {
        if (m_dirtyPosition) {
            beginGeometryChange();
            wl_subsurface::set_position(m_absolutePos.x(), m_absolutePos.y());
            m_dirtyPosition = false;
        }
    }

    if (m_node)
        m_node->preprocessed = false;
}

void QFbxDeclarativeVideoOutput::frameSwapped()
{
    endGeometryChange();
}

void QFbxDeclarativeVideoOutput::beginGeometryChange()
{
    if (!m_resizing && wl_subsurface::isInitialized())
        wl_subsurface::set_sync();

    m_resizing = true;
}

void QFbxDeclarativeVideoOutput::endGeometryChange()
{
    if (m_resizing && wl_subsurface::isInitialized())
        wl_subsurface::set_desync();

    m_resizing = false;
}

QT_END_NAMESPACE
