#include "qfbxmetadataprovider.h"
#include "qfbxplayersession.h"

#include <QtMultimedia/qmediametadata.h>
#include <QtCore/QStringList>
#include <QtCore/QSize>
#include <QtCore/QDateTime>

#include <fbxmms.h>

QT_BEGIN_NAMESPACE

QStringList QFbxMetaDataProvider::metadataKeys = QStringList()
    << QMediaMetaData::Author
    << QMediaMetaData::Comment
    << QMediaMetaData::Copyright
    << QMediaMetaData::Date
    << QMediaMetaData::Genre
    << QMediaMetaData::Title
    << QMediaMetaData::AlbumTitle
    << QMediaMetaData::AlbumArtist
    << QMediaMetaData::Composer
    << QMediaMetaData::LeadPerformer
    << QMediaMetaData::TrackNumber
    << QMediaMetaData::TrackCount;

QStringList QFbxMetaDataProvider::videoKeys = QStringList()
    << QMediaMetaData::VideoCodec
    << QMediaMetaData::Resolution
    << QMediaMetaData::PixelAspectRatio
    << QMediaMetaData::VideoFrameRate
    << QMediaMetaData::VideoBitRate;

QStringList QFbxMetaDataProvider::audioKeys = QStringList()
    << QMediaMetaData::AudioCodec
    << QMediaMetaData::ChannelCount
    << QMediaMetaData::SampleRate
    << QMediaMetaData::AudioBitRate
    << QMediaMetaData::AudioBitRate;

QFbxMetaDataProvider::QFbxMetaDataProvider(QFbxPlayerSession *session, QObject *parent)
    : QMetaDataReaderControl(parent)
    , m_session(session)
{
    fbxmms_register_media_video_tracks_changed(m_session->bus(),
            cb_video_tracks_changed, this);
    fbxmms_register_media_audio_tracks_changed(m_session->bus(),
            cb_audio_tracks_changed, this);
    fbxmms_register_media_metadata_changed(m_session->bus(),
            cb_metadata_changed, this);
}

QFbxMetaDataProvider::~QFbxMetaDataProvider()
{
    fbxmms_register_media_video_tracks_changed(m_session->bus(), NULL, NULL);
    fbxmms_register_media_audio_tracks_changed(m_session->bus(), NULL, NULL);
    fbxmms_register_media_metadata_changed(m_session->bus(), NULL, NULL);
}

void QFbxMetaDataProvider::cb_video_tracks_changed(void *priv, int media)
{
    QFbxMetaDataProvider *that = static_cast<QFbxMetaDataProvider *>(priv);

    if (media == that->m_session->id())
        that->updateMetadata(UpdateVideoTags);
}

void QFbxMetaDataProvider::cb_audio_tracks_changed(void *priv, int media)
{
    QFbxMetaDataProvider *that = static_cast<QFbxMetaDataProvider *>(priv);

    if (media == that->m_session->id())
        that->updateMetadata(UpdateAudioTags);
}

void QFbxMetaDataProvider::cb_metadata_changed(void *priv, int media)
{
    QFbxMetaDataProvider *that = static_cast<QFbxMetaDataProvider *>(priv);

    if (media == that->m_session->id())
        that->updateMetadata(UpdateMetadata);
}

bool QFbxMetaDataProvider::isMetaDataAvailable() const
{
    return !m_tags.isEmpty();
}

QVariant QFbxMetaDataProvider::metaData(const QString &key) const
{
    return m_tags.value(key);
}

QStringList QFbxMetaDataProvider::availableMetaData() const
{
    return m_tags.keys();
}

void QFbxMetaDataProvider::updateMetadata(QFbxMetaDataProvider::UpdateFlags flags)
{
    int ret;
    bool changed = false;
    QVariantMap oldTags = m_tags;

    if (flags & UpdateMetadata) {
        struct fbxmms_metadata m;

        foreach (QString key, metadataKeys)
            m_tags.remove(key);

        ret = fbxmms_media_metadata_get(m_session->bus(), m_session->id(), &m);
        if (ret == FBXMMS_SUCCESS) {
            if (*m.author)
                m_tags.insert(QMediaMetaData::Author,
                        QString::fromUtf8(m.author));

            if (*m.comment)
                m_tags.insert(QMediaMetaData::Comment,
                        QString::fromUtf8(m.comment));

            if (*m.copyright)
                m_tags.insert(QMediaMetaData::Copyright,
                        QString::fromUtf8(m.copyright));

            if (m.date)
                m_tags.insert(QMediaMetaData::Date,
                        QDateTime::fromMSecsSinceEpoch(m.date));

            if (*m.genre)
                m_tags.insert(QMediaMetaData::Genre,
                        QString::fromUtf8(m.genre));

            if (*m.title)
                m_tags.insert(QMediaMetaData::Title,
                        QString::fromUtf8(m.title));

            if (*m.album)
                m_tags.insert(QMediaMetaData::AlbumTitle,
                        QString::fromUtf8(m.album));

            if (*m.album_artist)
                m_tags.insert(QMediaMetaData::AlbumArtist,
                        QString::fromUtf8(m.album_artist));
            else if (*m.artist)
                m_tags.insert(QMediaMetaData::AlbumArtist,
                        QString::fromUtf8(m.artist));

            if (*m.composer)
                m_tags.insert(QMediaMetaData::Composer,
                        QString::fromUtf8(m.composer));

            if (*m.performer)
                m_tags.insert(QMediaMetaData::LeadPerformer,
                        QString::fromUtf8(m.performer));

            if (m.track_number > 0)
                m_tags.insert(QMediaMetaData::TrackNumber, m.track_number);

            if (m.track_total > 0)
                m_tags.insert(QMediaMetaData::TrackCount, m.track_total);

            fbxmms_free_struct_metadata(&m);
        }
    }

    if (flags & UpdateVideoTags) {
        struct fbxmms_video_track *vtracks;
        size_t size;

        foreach (QString key, videoKeys)
            m_tags.remove(key);

        ret = fbxmms_media_video_tracks_get(m_session->bus(), m_session->id(),
                &vtracks, &size);
        if (ret == FBXMMS_SUCCESS && size > 0) {
            struct fbxmms_video_track *t;
            int cur;

            ret = fbxmms_media_video_track_get(m_session->bus(),
                    m_session->id(), &cur);
            if (ret != FBXMMS_SUCCESS || cur < 0 || cur >= (int)size)
                cur = 0;

            t = &vtracks[cur];

            if (*t->codec)
                m_tags.insert(QMediaMetaData::VideoCodec,
                        QString::fromUtf8(t->codec));

            if (t->width > 0 || t->height > 0)
                m_tags.insert(QMediaMetaData::Resolution,
                        QSize(t->width, t->height));

            if (t->pixel_aspect_ratio.x > 0 && t->pixel_aspect_ratio.y > 0)
                m_tags.insert(QMediaMetaData::PixelAspectRatio,
                        QSize(t->pixel_aspect_ratio.x,
                              t->pixel_aspect_ratio.y));

            if (t->framerate.den > 0)
                m_tags.insert(QMediaMetaData::VideoFrameRate,
                        (qreal)t->framerate.num / (qreal)t->framerate.den);

            if (t->bitrate > 0)
                m_tags.insert(QMediaMetaData::VideoBitRate, (quint64)t->bitrate);

            for (size_t i = 0; i < size; i++)
                fbxmms_free_struct_video_track(&vtracks[i]);

            free(vtracks);
        }
    }

    if (flags & UpdateAudioTags) {
        struct fbxmms_audio_track *atracks;
        size_t size;

        foreach (QString key, audioKeys)
            m_tags.remove(key);

        ret = fbxmms_media_audio_tracks_get(m_session->bus(), m_session->id(),
                &atracks, &size);
        if (ret == FBXMMS_SUCCESS && size > 0) {
            struct fbxmms_audio_track *t;
            int cur;

            ret = fbxmms_media_audio_track_get(m_session->bus(),
                    m_session->id(), &cur);
            if (ret != FBXMMS_SUCCESS || cur < 0 || cur >= (int)size)
                cur = 0;

            t = &atracks[cur];

            if (*t->codec)
                m_tags.insert(QMediaMetaData::AudioCodec,
                        QString::fromUtf8(t->codec));

            if (t->channels > 0)
                m_tags.insert(QMediaMetaData::ChannelCount, t->channels);

            if (t->samplerate > 0)
                m_tags.insert(QMediaMetaData::SampleRate, t->samplerate);

            if (t->bitrate > 0)
                m_tags.insert(QMediaMetaData::AudioBitRate, (quint64)t->bitrate);

            if (*t->language)
                m_tags.insert(QMediaMetaData::Language,
                        QString::fromLatin1(t->language));

            for (size_t i = 0; i < size; i++)
                fbxmms_free_struct_audio_track(&atracks[i]);

            free(atracks);
        }
    }

    QMapIterator<QString, QVariant> i(m_tags);
    while (i.hasNext()) {
         i.next();
         if (i.value() != oldTags.value(i.key())) {
             changed = true;
             emit metaDataChanged(i.key(), i.value());
         }
    }

    if (changed)
        emit metaDataChanged();

    if (oldTags.isEmpty() != m_tags.isEmpty())
        emit metaDataAvailableChanged(!m_tags.isEmpty());
}

QT_END_NAMESPACE
