#include <fbxmms.h>

#include "qfbxplayercontrol.h"
#include "qfbxplayersession.h"

#include <private/qmediaplaylistnavigator_p.h>

QT_BEGIN_NAMESPACE

QFbxPlayerControl::QFbxPlayerControl(QFbxPlayerSession *session, QObject *parent)
    : QMediaPlayerControl(parent)
    , m_session(session)
    , m_state(QMediaPlayer::StoppedState)
    , m_mediaStatus(QMediaPlayer::NoMedia)
    , m_volume(100)
    , m_mute(false)
{
    fbxmms_register_media_error(m_session->bus(), cb_error, this);
    fbxmms_register_media_state(m_session->bus(), cb_media_state_changed, this);
    fbxmms_register_media_playback_state(m_session->bus(),
            cb_playback_state_changed, this);
}

QFbxPlayerControl::~QFbxPlayerControl()
{
    fbxmms_register_media_playback_state(m_session->bus(), NULL, NULL);
    fbxmms_register_media_state(m_session->bus(), NULL, NULL);
    fbxmms_register_media_error(m_session->bus(), NULL, NULL);
    fbxmms_media_close(m_session->bus(), m_session->id());
}

static QMediaPlayer::MediaStatus
convert_media_state(enum fbxmms_media_state state)
{
    switch (state) {
    case FBXMMS_MEDIA_STATE_NOMEDIA:
        return QMediaPlayer::NoMedia;
    case FBXMMS_MEDIA_STATE_LOADING:
        return QMediaPlayer::LoadingMedia;
    case FBXMMS_MEDIA_STATE_BUFFERING:
        return QMediaPlayer::StalledMedia;
    case FBXMMS_MEDIA_STATE_READY:
        return QMediaPlayer::BufferedMedia;
    case FBXMMS_MEDIA_STATE_ERROR:
        return QMediaPlayer::InvalidMedia;
    case FBXMMS_MEDIA_STATE_ENDED:
        return QMediaPlayer::EndOfMedia;
    default:
        return QMediaPlayer::UnknownMediaStatus;
    }
}

static QMediaPlayer::State
convert_playback_state(enum fbxmms_playback_state state)
{
    switch (state) {
    default:
    case FBXMMS_PLAYBACK_STATE_STOPPED:
        return QMediaPlayer::StoppedState;
    case FBXMMS_PLAYBACK_STATE_PAUSED:
        return QMediaPlayer::PausedState;
    case FBXMMS_PLAYBACK_STATE_PLAYING:
        return QMediaPlayer::PlayingState;
    }
}

void QFbxPlayerControl::cb_error(void *priv, int media,
        const struct fbxmms_media_error *error)
{
    QFbxPlayerControl *control = static_cast<QFbxPlayerControl *>(priv);

    if (media == control->m_session->id())
        emit control->error(QMediaPlayer::ResourceError, tr("Media error"));
}

void QFbxPlayerControl::cb_media_state_changed(void *priv, int media,
    enum fbxmms_media_state state)
{
    QFbxPlayerControl *control = static_cast<QFbxPlayerControl *>(priv);

    if (media == control->m_session->id())
        control->setMediaStatus(convert_media_state(state));
}

void QFbxPlayerControl::cb_playback_state_changed(void *priv, int media,
    enum fbxmms_playback_state prev,
    enum fbxmms_playback_state state,
    enum fbxmms_playback_state pending)
{
    QFbxPlayerControl *control = static_cast<QFbxPlayerControl *>(priv);

    if (media == control->m_session->id())
        control->setState(convert_playback_state(state));
}

QMediaContent QFbxPlayerControl::media() const
{
    char *url;
    int ret;

    ret = fbxmms_media_url_get(m_session->bus(), m_session->id(), &url);
    if (ret != FBXMMS_SUCCESS)
        return QMediaContent();

    return QMediaContent(QUrl::fromEncoded(url));
}

const QIODevice *QFbxPlayerControl::mediaStream() const
{
    return 0;
}

void QFbxPlayerControl::setMedia(const QMediaContent &content, QIODevice *stream)
{
    enum fbxmms_audio_role audioRole;

    if (content.isNull()) {
        fbxmms_media_close(m_session->bus(), m_session->id());
        return;
    }

    fbxmms_media_open(m_session->bus(), m_session->id(),
            content.canonicalUrl().toEncoded().constData());

    switch (m_session->audioRole()) {
    default:
        audioRole = FBXMMS_AUDIO_ROLE_UNKNOWN;
        break;
    case QAudio::MusicRole:
        audioRole = FBXMMS_AUDIO_ROLE_MUSIC;
        break;
    case QAudio::VideoRole:
        audioRole = FBXMMS_AUDIO_ROLE_VIDEO;
        break;
    case QAudio::NotificationRole:
    case QAudio::AccessibilityRole:
    case QAudio::SonificationRole:
        audioRole = FBXMMS_AUDIO_ROLE_EVENT;
        break;
    case QAudio::GameRole:
        audioRole = FBXMMS_AUDIO_ROLE_GAME;
        break;
    }

    fbxmms_media_audio_role_set(m_session->bus(), m_session->id(), audioRole);

    setVolume(m_volume);
}

qint64 QFbxPlayerControl::position() const
{
    int64_t position = 0;

    fbxmms_media_position_get(m_session->bus(), m_session->id(),
            FBXMMS_UNIT_TIME, &position);

    return position;
}

qint64 QFbxPlayerControl::duration() const
{
    int32_t duration = 0;

    fbxmms_media_duration_get(m_session->bus(), m_session->id(), &duration);

    return duration;
}

void QFbxPlayerControl::setState(QMediaPlayer::State newState)
{
    QMediaPlayer::State prevState;

    prevState = state();
    m_state = newState;
    newState = state();

    if (newState != prevState)
        emit stateChanged(newState);
}

QMediaPlayer::State QFbxPlayerControl::state() const
{
    if (m_mediaStatus == QMediaPlayer::EndOfMedia)
        return QMediaPlayer::StoppedState;
    else
        return m_state;
}

void QFbxPlayerControl::setMediaStatus(QMediaPlayer::MediaStatus status)
{
    if (status != m_mediaStatus) {
        QMediaPlayer::State prevState, curState;

        prevState = state();
        m_mediaStatus = status;
        emit mediaStatusChanged(status);
        curState = state();

        if (curState != prevState)
            emit stateChanged(curState);
    }
}

QMediaPlayer::MediaStatus QFbxPlayerControl::mediaStatus() const
{
    return m_mediaStatus;
}

int QFbxPlayerControl::bufferStatus() const
{
    return 100;
}

bool QFbxPlayerControl::isSeekable() const
{
    return true;
}

QMediaTimeRange QFbxPlayerControl::availablePlaybackRanges() const
{
    return QMediaTimeRange(0, duration());
}

qreal QFbxPlayerControl::playbackRate() const
{
    int speed = 100;

    fbxmms_media_speed_get(m_session->bus(), m_session->id(), &speed);

    return speed / 100.0;
}

void QFbxPlayerControl::setPlaybackRate(qreal rate)
{
    int ret;

    ret = fbxmms_media_speed_set(m_session->bus(), m_session->id(),
            rate * 100.0);
    if (ret == FBXMMS_SUCCESS)
        emit playbackRateChanged(((int)(rate * 100)) / 100.0);
}

void QFbxPlayerControl::setPosition(qint64 pos)
{
    fbxmms_media_seek(m_session->bus(), m_session->id(), pos,
            FBXMMS_UNIT_TIME, FBXMMS_SEEK_SET);
}

void QFbxPlayerControl::play()
{
    fbxmms_media_play(m_session->bus(), m_session->id());
    if (m_mediaStatus == QMediaPlayer::EndOfMedia)
        setPosition(0);
}

void QFbxPlayerControl::pause()
{
    fbxmms_media_pause(m_session->bus(), m_session->id());
    if (m_mediaStatus == QMediaPlayer::EndOfMedia)
        setPosition(0);
}

void QFbxPlayerControl::stop()
{
    fbxmms_media_stop(m_session->bus(), m_session->id());
}

int QFbxPlayerControl::volume() const
{
    return m_volume;
}

void QFbxPlayerControl::setVolume(int volume)
{
    m_volume = volume;
    fbxmms_media_volume_set(m_session->bus(), m_session->id(), volume);
}

bool QFbxPlayerControl::isMuted() const
{
    return m_mute;
}

void QFbxPlayerControl::setMuted(bool muted)
{
    m_mute = muted;
    fbxmms_media_mute_set(m_session->bus(), m_session->id(), muted);
}

void QFbxPlayerControl::setVideoOutput(QObject *output)
{
}

bool QFbxPlayerControl::isAudioAvailable() const
{
    return m_session->id() != 1;
}

bool QFbxPlayerControl::isVideoAvailable() const
{
    return m_session->id() < 2;
}

QT_END_NAMESPACE
