#include "qfbxdeclarativevideooutput.h"
#include "foreign_surface_manager.h"
#include "qdeclarativevideooutput_p.h"

#include <QtGui/qguiapplication.h>
#include <QtGui/qpa/qplatformnativeinterface.h>
#include <QtQuick/qquickwindow.h>
#include <QtQuick/qsgsimplerectnode.h>
#include <QtMultimedia/qmediaservice.h>
#include <QtMultimedia/qvideowindowcontrol.h>
#include <QDebug>

QT_BEGIN_NAMESPACE

QFbxDeclarativeVideoOutput::QFbxDeclarativeVideoOutput(QDeclarativeVideoOutput *parent)
    : QDeclarativeVideoBackend(parent)
    , m_videoWindowControl(0)
    , m_manager(0)
    , m_window(0)
    , m_resizing(false)
{
}

QFbxDeclarativeVideoOutput::~QFbxDeclarativeVideoOutput()
{
    unbind();
    releaseSource();
    releaseControl();
}

bool QFbxDeclarativeVideoOutput::init(QMediaService *service)
{
    QMediaControl *control = service->requestControl(QVideoWindowControl_iid);
    if (!control)
        return false;

    m_videoWindowControl = qobject_cast<QVideoWindowControl *>(control);
    if (!m_videoWindowControl)
        return false;

    m_service = service;
    QObject::connect(m_videoWindowControl.data(),
                     SIGNAL(nativeSizeChanged()),
                     q, SLOT(_q_updateNativeSize()));

    m_manager = fbx::ForeignSurfaceManager::getInstance();
    connect(m_manager, &fbx::ForeignSurfaceManager::connected,
            this, &QFbxDeclarativeVideoOutput::bind);

    bind();

    q->setFlag(QQuickItem::ItemHasContents, true);

    return true;
}

void QFbxDeclarativeVideoOutput::itemChange(QQuickItem::ItemChange change,
        const QQuickItem::ItemChangeData &changeData)
{
    switch (change) {
    case QQuickItem::ItemVisibleHasChanged:
    case QQuickItem::ItemSceneChange:
        rebind();
        break;
    default:
        break;
    }
}

void QFbxDeclarativeVideoOutput::releaseSource()
{
}

void QFbxDeclarativeVideoOutput::releaseControl()
{
    if (m_videoWindowControl) {
        if (m_service)
            m_service->releaseControl(m_videoWindowControl);
        m_videoWindowControl = 0;
    }
}

QSize QFbxDeclarativeVideoOutput::nativeSize() const
{
    return m_videoWindowControl->nativeSize();
}

void QFbxDeclarativeVideoOutput::updateGeometry()
{
    switch (q->fillMode()) {
    case QDeclarativeVideoOutput::PreserveAspectFit:
        m_videoWindowControl->setAspectRatioMode(Qt::KeepAspectRatio);
        break;
    case QDeclarativeVideoOutput::PreserveAspectCrop:
        m_videoWindowControl->setAspectRatioMode(Qt::KeepAspectRatioByExpanding);
        break;
    case QDeclarativeVideoOutput::Stretch:
        m_videoWindowControl->setAspectRatioMode(Qt::IgnoreAspectRatio);
        break;
    };
}

QSGNode *QFbxDeclarativeVideoOutput::updatePaintNode(QSGNode *node,
        QQuickItem::UpdatePaintNodeData *data)
{
    Q_UNUSED(data);

    QSGSimpleRectNode *rectangle = static_cast<QSGSimpleRectNode *>(node);
    if (!rectangle) {
        rectangle = new QSGSimpleRectNode;
        rectangle->setColor(Qt::transparent);
        rectangle->material()->setFlag(QSGMaterial::Blending, false);
    }

    rectangle->setRect(q->boundingRect());

    return rectangle;
}

QAbstractVideoSurface *QFbxDeclarativeVideoOutput::videoSurface() const
{
    return 0;
}

QRectF QFbxDeclarativeVideoOutput::adjustedViewport() const
{
    return QRectF(QPointF(0, 0), nativeSize());
}

void QFbxDeclarativeVideoOutput::unbind()
{
    m_resizing = false;

    if (m_window) {
        disconnect(m_window, &QQuickWindow::beforeSynchronizing,
                   this, &QFbxDeclarativeVideoOutput::sync);
        disconnect(m_window, &QQuickWindow::frameSwapped,
                   this, &QFbxDeclarativeVideoOutput::frameSwapped);
        m_window = 0;
    }

    if (wl_subsurface::isInitialized())
        wl_subsurface::destroy();
}

void QFbxDeclarativeVideoOutput::rebind()
{
    unbind();
    bind();
}

void QFbxDeclarativeVideoOutput::bind()
{
    if (wl_subsurface::isInitialized())
        return;

    if (!q->isVisible())
        return;

    if (!m_manager->isConnected())
        return;

    WId id = m_videoWindowControl->winId();
    if (!id)
        return;

    struct ::wl_surface *parent = 0;

    QQuickWindow *w = q->window();
    if (w) {
        w->create();
        QPlatformNativeInterface *native
            = QGuiApplication::platformNativeInterface();
        parent = static_cast<struct wl_surface *>(
            native->nativeResourceForWindow("surface", w));
    }

    if (!parent)
        return;

    m_window = w;
    if (m_window) {
        connect(m_window, &QQuickWindow::beforeSynchronizing,
                this, &QFbxDeclarativeVideoOutput::sync, Qt::DirectConnection);
        connect(m_window, &QQuickWindow::frameSwapped,
                this, &QFbxDeclarativeVideoOutput::frameSwapped, Qt::DirectConnection);
    }

    m_absolutePos = QPoint(0, 0);

    struct ::wl_surface *foreign_surface = m_manager->import_surface(id);
    wl_subsurface::init(m_manager->get_subsurface(foreign_surface, parent));
    wl_subsurface::place_below(parent);
    wl_surface_destroy(foreign_surface);

    m_resizing = true;
}

void QFbxDeclarativeVideoOutput::frameSwapped()
{
    endGeometryChange();
}

QRectF QFbxDeclarativeVideoOutput::absoluteGeometry() const
{
    const qreal devicePixelRatio =
        (q->window() ? q->window()->effectiveDevicePixelRatio() : qApp->devicePixelRatio());

    QRectF rect = q->mapRectToScene(QRectF(0, 0, q->width(), q->height()));

    return QRectF(rect.topLeft() * devicePixelRatio,
                  rect.size() * devicePixelRatio);
}

void QFbxDeclarativeVideoOutput::sync()
{
    QRect rect = absoluteGeometry().toRect();

    if (wl_subsurface::isInitialized()) {
        if (rect.topLeft() != m_absolutePos) {
            m_absolutePos = rect.topLeft();
            beginGeometryChange();
            wl_subsurface::set_position(m_absolutePos.x(), m_absolutePos.y());
        }
    }

    if (rect.size() != m_size) {
        m_size = rect.size();
        beginGeometryChange();
        m_videoWindowControl->setDisplayRect(QRect(QPoint(0, 0), m_size));
    }
}

void QFbxDeclarativeVideoOutput::beginGeometryChange()
{
    if (!m_resizing && wl_subsurface::isInitialized()) {
        m_resizing = true;
        wl_subsurface::set_sync();
    }
}

void QFbxDeclarativeVideoOutput::endGeometryChange()
{
    if (m_resizing && wl_subsurface::isInitialized())
        wl_subsurface::set_desync();

    m_resizing = false;
}

QT_END_NAMESPACE
