#include "qeglfshooks.h"

#include <QtDebug>

#include <QtPlatformSupport/private/qeglconvenience_p.h>
#include <QtPlatformSupport/private/qeglplatformcontext_p.h>

#include <libgdl.h>

QT_BEGIN_NAMESPACE

#define NATIVE_WINDOW_TO_PLANE(id) ((gdl_plane_id_t)(uintptr_t)(id))
#define NATIVE_WINDOW_FROM_PLANE(plane) ((EGLNativeWindowType)plane)
#define NATIVE_WINDOW_INVALID ((EGLNativeWindowType)-1)

static EGLNativeWindowType initPlane(gdl_plane_id_t plane_id, gdl_pixel_format_t format, const QSize &size)
{
    gdl_color_space_t cspace;
    gdl_rectangle_t rect;
    gdl_boolean_t premul;
    gdl_boolean_t abort;
    gdl_ret_t rc;

    if (plane_id == GDL_PLANE_ID_UNDEFINED)
        goto fail;

    switch (format) {
    case GDL_PF_ARGB_32:
    case GDL_PF_ARGB_16_1555:
        cspace = GDL_COLOR_SPACE_RGB;
        premul = GDL_TRUE;
        break;
    case GDL_PF_RGB_32:
    case GDL_PF_RGB_16:
        cspace = GDL_COLOR_SPACE_RGB;
        premul = GDL_FALSE;
        break;
    default:
        qWarning("invalid gdl format %u for EGLWindow", format);
        goto fail;
    }

    gdl_plane_reset(plane_id);

    abort = GDL_TRUE;

    rc = gdl_plane_config_begin(plane_id);
    if (rc != GDL_SUCCESS)
        goto commit;

    rc = gdl_plane_set_uint(GDL_PLANE_SRC_COLOR_SPACE, cspace);
    if (rc != GDL_SUCCESS)
        goto commit;

    rc = gdl_plane_set_uint(GDL_PLANE_PIXEL_FORMAT, format);
    if (rc != GDL_SUCCESS)
        goto commit;

    rc = gdl_plane_set_uint(GDL_PLANE_ALPHA_PREMULT, premul);
    if (rc != GDL_SUCCESS)
        goto commit;

    rect.origin.x = 0;
    rect.origin.y = 0;
    rect.width = size.width();
    rect.height = size.height();

    rc = gdl_plane_set_rect(GDL_PLANE_DST_RECT, &rect);
    if (rc != GDL_SUCCESS)
        goto commit;

    rc = gdl_plane_set_rect(GDL_PLANE_SRC_RECT, &rect);
    if (rc != GDL_SUCCESS)
        goto commit;

    abort = GDL_FALSE;

commit:
    rc = gdl_plane_config_end(abort);
    if (abort || rc != GDL_SUCCESS) {
        qWarning("failed to configure gdl plane");
        goto fail;
    }

    return NATIVE_WINDOW_FROM_PLANE(plane_id);

fail:
    return NATIVE_WINDOW_INVALID;
}

class QEglFSIceHooks : public QEglFSHooks
{
public:
    QEglFSIceHooks() {
        for (int i = 0; i < 4; i++)
            m_plane_available[i] = true;

        reservePlane(GDL_PLANE_ID_UPP_D);
    }

    virtual void platformInit();
    virtual void platformDestroy();
    virtual EGLNativeDisplayType platformDisplay() const;
    virtual QSize screenSize() const;
    virtual int screenDepth() const;
    virtual EGLNativeWindowType createNativeWindow(QPlatformWindow *platformWindow, const QSize &size, const QSurfaceFormat &format);
    virtual void destroyNativeWindow(EGLNativeWindowType window);
    virtual bool hasCapability(QPlatformIntegration::Capability cap) const;

private:
    gdl_plane_id_t reservePlane(gdl_plane_id_t plane = GDL_PLANE_ID_UNDEFINED) const;
    void releasePlane(gdl_plane_id_t plane);

    mutable bool m_plane_available[4];
};

void QEglFSIceHooks::platformInit()
{
    gdl_init(0);
}

EGLNativeDisplayType QEglFSIceHooks::platformDisplay() const
{
    return EGL_DEFAULT_DISPLAY;
}

void QEglFSIceHooks::platformDestroy()
{
    gdl_close();
}

QSize QEglFSIceHooks::screenSize() const
{
    gdl_display_info_t disp;

    if (gdl_get_display_info(GDL_DISPLAY_ID_0, &disp) != GDL_SUCCESS) {
        qWarning("failed to get gdl display info");
        return QSize();
    }

    return QSize(disp.tvmode.width, disp.tvmode.height);
}

int QEglFSIceHooks::screenDepth() const
{
    return 32;
}

EGLNativeWindowType QEglFSIceHooks::createNativeWindow(QPlatformWindow *platformWindow, const QSize &size, const QSurfaceFormat &format)
{
    gdl_plane_id_t plane = reservePlane();
    if (plane == GDL_PLANE_ID_UNDEFINED)
        qWarning("no gdl plane available to create window");

    gdl_pixel_format_t pixfmt = GDL_PF_COUNT;
    int r = format.redBufferSize();
    int g = format.greenBufferSize();
    int b = format.blueBufferSize();
    int a = format.alphaBufferSize();

    qDebug() << "creating window with format" << format;

    if (r == 8 && g == 8 && b == 8 && a == 8)
        pixfmt = GDL_PF_ARGB_32;
    else if (r == 8 && g == 8 && b == 8 && a == 0)
        pixfmt = GDL_PF_RGB_32;
    else if (r == 5 && g == 6 && b == 5 && a == 0)
        pixfmt = GDL_PF_RGB_16;
    else if (r == 5 && g == 5 && b == 5 && a == 1)
        pixfmt = GDL_PF_ARGB_16_1555;

    EGLNativeWindowType window = initPlane(plane, pixfmt, size);
    if (window == NATIVE_WINDOW_INVALID)
        releasePlane(plane);

    return window;
}

void QEglFSIceHooks::destroyNativeWindow(EGLNativeWindowType window)
{
    releasePlane(NATIVE_WINDOW_TO_PLANE(window));
}

bool QEglFSIceHooks::hasCapability(QPlatformIntegration::Capability cap) const
{
    switch (cap) {
        case QPlatformIntegration::ThreadedPixmaps:
        case QPlatformIntegration::OpenGL:
        case QPlatformIntegration::ThreadedOpenGL:
        case QPlatformIntegration::BufferQueueingOpenGL:
            return true;
        default:
            return false;
    }
}

gdl_plane_id_t QEglFSIceHooks::reservePlane(gdl_plane_id_t plane) const
{
    if (plane != GDL_PLANE_ID_UNDEFINED) {
        if (m_plane_available[plane - GDL_PLANE_ID_UPP_A])
            return plane;
    } else {
        for (int i = 0; i < 4; i++) {
            if (m_plane_available[i]) {
                m_plane_available[i] = false;
                return (gdl_plane_id_t)(GDL_PLANE_ID_UPP_A + i);
            }
        }
    }

    return GDL_PLANE_ID_UNDEFINED;
}

void QEglFSIceHooks::releasePlane(gdl_plane_id_t plane)
{
    m_plane_available[plane - GDL_PLANE_ID_UPP_A] = true;
}

QEglFSIceHooks eglFSIceHooks;
QEglFSHooks *platformHooks = &eglFSIceHooks;

QT_END_NAMESPACE
