/* GStreamer
 * Copyright (C) <2012> Freebox SAS (contact <avrac@freebox.fr>)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <arpa/inet.h>

#include <gst/rtp/gstrtpbuffer.h>
#include "gstrtpelements.h"
#include "gstrtpalacdepay.h"
#include "gstrtputils.h"

#define ATOM_TAG_ALAC 0x616c6163

struct _AlacAtom
{
  guint32 atom_size;
  guint32 tag;
  guint32 tag_version;
  guint32 samples_per_frame;
  guint8 compatible_version;
  guint8 sample_size;
  guint8 history_mult;
  guint8 initial_history;
  guint8 kmodifier;
  guint8 channels;
  guint16 max_run;
  guint32 max_coded_frame_size;
  guint32 average_bitrate;
  guint32 sample_rate;
} __attribute__ ((__packed__));

typedef struct _AlacAtom AlacAtom;

GST_DEBUG_CATEGORY_STATIC (rtpalacdepay_debug);
#define GST_CAT_DEFAULT (rtpalacdepay_debug)

static GstStaticPadTemplate gst_rtp_alac_depay_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-alac"));

static GstStaticPadTemplate gst_rtp_alac_depay_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-rtp, "
        "media = (string) audio, "
        "payload = (int) " GST_RTP_PAYLOAD_DYNAMIC_STRING ", "
        "clock-rate = (int) [ 1, MAX ], "
        "encoding-name = (string) APPLELOSSLESS")
    );

#define gst_rtp_alac_depay_parent_class parent_class
G_DEFINE_TYPE (GstRtpALACDepay, gst_rtp_alac_depay,
    GST_TYPE_RTP_BASE_DEPAYLOAD);
GST_ELEMENT_REGISTER_DEFINE_WITH_CODE (rtpalacdepay, "rtpalacdepay",
    GST_RANK_SECONDARY, GST_TYPE_RTP_ALAC_DEPAY, rtp_element_init (plugin));

static gboolean gst_rtp_alac_depay_setcaps (GstRTPBaseDepayload * depayload,
    GstCaps * caps);
static GstBuffer *gst_rtp_alac_depay_process (GstRTPBaseDepayload * depayload,
    GstRTPBuffer * rtp);

static void
gst_rtp_alac_depay_class_init (GstRtpALACDepayClass * klass)
{
  GstElementClass *gstelement_class;
  GstRTPBaseDepayloadClass *gstrtpbasedepayload_class;

  gstelement_class = (GstElementClass *) klass;
  gstrtpbasedepayload_class = (GstRTPBaseDepayloadClass *) klass;

  gst_element_class_add_static_pad_template (gstelement_class,
      &gst_rtp_alac_depay_src_template);
  gst_element_class_add_static_pad_template (gstelement_class,
      &gst_rtp_alac_depay_sink_template);

  gst_element_class_set_static_metadata (gstelement_class,
      "RTP ALAC depayloader", "Codec/Depayloader/Network/RTP",
      "Extracts ALAC (Apple Lossless Audio Codec) audio from RTP packets",
      "Arnaud Vrac <avrac@freebox.fr>");

  gstrtpbasedepayload_class->set_caps = gst_rtp_alac_depay_setcaps;
  gstrtpbasedepayload_class->process_rtp_packet = gst_rtp_alac_depay_process;

  GST_DEBUG_CATEGORY_INIT (rtpalacdepay_debug, "rtpalacdepay", 0,
      "ALAC Audio RTP Depayloader");
}

static void
gst_rtp_alac_depay_init (GstRtpALACDepay * rtpalacdepay)
{
}

static gboolean
gst_rtp_alac_depay_setcaps (GstRTPBaseDepayload * depayload, GstCaps * caps)
{
  GstStructure *structure;
  GstCaps *srccaps;
  const gchar *str;
  gint clock_rate;
  gboolean res;

  structure = gst_caps_get_structure (caps, 0);

  if (!gst_structure_get_int (structure, "clock-rate", &clock_rate))
    clock_rate = 90000;
  depayload->clock_rate = clock_rate;

  srccaps = gst_caps_new_empty_simple ("audio/x-alac");

  if ((str = gst_structure_get_string (structure, "format-parameters"))) {
    GstBuffer *buffer;
    guint32 samples_per_frame;
    guint8 compatible_version;
    guint8 sample_size;
    guint8 history_mult;
    guint8 initial_history;
    guint8 kmodifier;
    guint8 channels;
    guint16 max_run;
    guint32 max_coded_frame_size;
    guint32 average_bitrate;
    guint32 sample_rate;
    AlacAtom atom;
    guint size;

    if (sscanf (str, "%u %hhu %hhu %hhu %hhu %hhu %hhu %hu %u %u %u",
            &samples_per_frame, &compatible_version, &sample_size,
            &history_mult, &initial_history, &kmodifier, &channels, &max_run,
            &max_coded_frame_size, &average_bitrate, &sample_rate) != 11)
      goto bad_parameters;

    size = sizeof (atom);

    atom.atom_size = htonl (size);
    atom.tag = htonl (ATOM_TAG_ALAC);
    atom.tag_version = htonl (0);

    atom.samples_per_frame = htonl (samples_per_frame);
    atom.compatible_version = compatible_version;
    atom.sample_size = sample_size;
    atom.history_mult = history_mult;
    atom.initial_history = initial_history;
    atom.kmodifier = kmodifier;
    atom.channels = channels;
    atom.max_run = htons (max_run);
    atom.max_coded_frame_size = htonl (max_coded_frame_size);
    atom.average_bitrate = htonl (average_bitrate);
    atom.sample_rate = htonl (sample_rate);

    buffer = gst_buffer_new_wrapped (g_memdup (&atom, size), size);

    gst_caps_set_simple (srccaps,
        "channels", G_TYPE_INT, (gint) channels,
        "rate", G_TYPE_INT, (gint) sample_rate,
        "codec_data", GST_TYPE_BUFFER, buffer, NULL);

    gst_buffer_unref (buffer);
  }

bad_parameters:
  res = gst_pad_set_caps (depayload->srcpad, srccaps);
  gst_caps_unref (srccaps);

  return res;
}

static GstBuffer *
gst_rtp_alac_depay_process (GstRTPBaseDepayload * depayload, GstRTPBuffer * rtp)
{
  GstBuffer *outbuf;

  outbuf = gst_rtp_buffer_get_payload_buffer (rtp);

  gst_rtp_drop_non_audio_meta (depayload, outbuf);

  return outbuf;
}
