#include <gst/base/gstbytereader.h>
#include <gst/base/gstbytewriter.h>

#include "gstsubsamplepatcher.h"

static gboolean
patcher_init (GstSubsamplePatcher * patcher, gboolean all_protected,
    gssize au_size, GstBuffer * subsamples_buffer)
{
  GstMapInfo mapinfo;
  GstByteReader br;
  guint16 clear;
  guint32 prot;
  guint i;

  if (all_protected) {
    if (au_size < 0)
      return FALSE;

    patcher->src.subsample_count = 1;
    patcher->src.subsamples[0].clear_bytes = 0;
    patcher->src.subsamples[0].protected_bytes = au_size;
    return TRUE;
  }

  if (!gst_buffer_map (subsamples_buffer, &mapinfo, GST_MAP_READ))
    return FALSE;

  if (mapinfo.size % 6)
    return FALSE;

  patcher->src.subsample_count = mapinfo.size / 6;

  if (patcher->src.subsample_count > GST_MAX_SUBSAMPLE_PER_AU)
    return FALSE;

  gst_byte_reader_init (&br, mapinfo.data, mapinfo.size);

  for (i = 0; i < patcher->src.subsample_count; ++i) {
    clear = 0;
    gst_byte_reader_get_uint16_be (&br, &clear);
    patcher->src.subsamples[i].clear_bytes = clear;

    prot = 0;
    gst_byte_reader_get_uint32_be (&br, &prot);
    patcher->src.subsamples[i].protected_bytes = prot;
  }

  gst_buffer_unmap (subsamples_buffer, &mapinfo);

  return TRUE;
}

gboolean
gst_subsample_patcher_init (GstSubsamplePatcher * patcher, GstBuffer * buffer)
{
  GstProtectionMeta *prot_meta;
  GstBuffer *subsamples_buffer;
  gboolean encrypted;
  const GValue *v;
  unsigned count;

  patcher->init = FALSE;

  if (!buffer)
    return TRUE;

  prot_meta = gst_buffer_get_protection_meta (buffer);
  if (!prot_meta)
    return TRUE;

  patcher->src.subsample_count = 0;
  patcher->dst.subsample_count = 0;
  patcher->pos = 0;
  patcher->error = FALSE;

  if (!gst_structure_get_boolean (prot_meta->info, "encrypted", &encrypted))
    encrypted = FALSE;

  if (!encrypted)
    return TRUE;

  if (!gst_structure_get_uint (prot_meta->info, "subsample_count", &count))
    count = 0;

  if (count == 0) {
    if (!patcher_init (patcher, TRUE, gst_buffer_get_size (buffer), NULL))
      return FALSE;

    patcher->init = TRUE;
    return TRUE;
  }

  v = gst_structure_get_value (prot_meta->info, "subsamples");
  if (!v)
    return FALSE;

  subsamples_buffer = gst_value_get_buffer (v);
  if (!subsamples_buffer)
    return FALSE;

  if (!patcher_init (patcher, FALSE, -1, subsamples_buffer))
    return FALSE;

  patcher->init = TRUE;
  return TRUE;
}

gboolean
gst_subsample_patcher_skip (GstSubsamplePatcher * patcher, gsize count)
{
  if (!patcher->init)
    return TRUE;

  if (patcher->error)
    return FALSE;

  if (count == 0)
    return TRUE;

  patcher->pos += count;

  return TRUE;
}

static GstSubsample *
find_subsample (GstSubsampleMap * subsample_map, gsize offset)
{
  gsize pos = 0;

  for (guint i = 0; i < subsample_map->subsample_count; i++) {
    GstSubsample *subsample = &subsample_map->subsamples[i];
    pos += subsample->clear_bytes;
    if (offset <= pos)
      return subsample;

    pos += subsample->protected_bytes;
    if (offset < pos) {
      /* inserting in the middle of protected data is not supported */
      return NULL;
    }
  }

  return NULL;
}

static gboolean
insert_clear_subsample (GstSubsampleMap * subsample_map,
    gsize offset, gsize count)
{
  GstSubsample *subsample;

  if (count == 0)
    return TRUE;

  subsample = find_subsample(subsample_map, offset);
  if (!subsample)
    return FALSE;

  subsample->clear_bytes += count;

  return TRUE;
}

static GstSubsample *
last_subsample (GstSubsampleMap * subsample_map)
{
  if (subsample_map->subsample_count > 0)
    return &subsample_map->subsamples[subsample_map->subsample_count - 1];
  else
    return NULL;
}

static gboolean
append_clear_subsample (GstSubsampleMap * subsample_map, gsize count)
{
  GstSubsample *last;

  if (count == 0)
    return TRUE;

  last = last_subsample (subsample_map);

  if (last && (last->protected_bytes == 0)) {
    last->clear_bytes += count;
    return TRUE;
  }

  if ((subsample_map->subsample_count + 1) > GST_MAX_SUBSAMPLE_PER_AU)
    return FALSE;

  subsample_map->subsample_count++;

  last = last_subsample (subsample_map);

  last->clear_bytes = count;
  last->protected_bytes = 0;

  return TRUE;
}

static gboolean
append_protected_subsample (GstSubsampleMap * subsample_map, gsize count)
{
  GstSubsample *last;

  if (count == 0)
    return TRUE;

  last = last_subsample (subsample_map);

  if (last) {
    last->protected_bytes += count;
    return TRUE;
  }

  if ((subsample_map->subsample_count + 1) > GST_MAX_SUBSAMPLE_PER_AU)
    return FALSE;

  subsample_map->subsample_count++;

  last = last_subsample (subsample_map);

  last->clear_bytes = 0;
  last->protected_bytes = count;

  return TRUE;
}

gboolean
gst_subsample_patcher_insert (GstSubsamplePatcher * patcher, gsize count)
{
  if (!patcher->init)
    return TRUE;

  if (patcher->error)
    return FALSE;

  if (count == 0)
    return TRUE;

  if (!append_clear_subsample (&patcher->dst, count)) {
    patcher->error = TRUE;
    return FALSE;
  }

  return TRUE;
}

gboolean
gst_subsample_patcher_insert_at (GstSubsamplePatcher * patcher,
    gsize offset, gsize count)
{
  if (!patcher->init)
    return TRUE;

  if (patcher->error)
    return FALSE;

  if (count == 0)
    return TRUE;

  if (!insert_clear_subsample (&patcher->dst, offset, count)) {
    patcher->error = TRUE;
    return FALSE;
  }

  return TRUE;
}

gboolean
gst_subsample_patcher_copy (GstSubsamplePatcher * patcher, gsize count)
{
  const GstSubsample *begin;
  const GstSubsample *end;
  const GstSubsample *it;
  GstSubsample c;
  gsize skip;
  gsize op;

  if (!patcher->init)
    return TRUE;

  if (patcher->error)
    return FALSE;

  if (count == 0)
    return TRUE;

  skip = patcher->pos;

  begin = &patcher->src.subsamples[0];
  end = &patcher->src.subsamples[patcher->src.subsample_count];

  for (it = begin; it < end; ++it) {
    c = *it;

    op = MIN (c.clear_bytes, skip);
    skip -= op;
    c.clear_bytes -= op;

    op = MIN (c.protected_bytes, skip);
    skip -= op;
    c.protected_bytes -= op;

    if (skip > 0)
      continue;

    op = MIN (c.clear_bytes, count);
    count -= op;
    c.clear_bytes -= op;

    if (!append_clear_subsample (&patcher->dst, op)) {
      patcher->error = TRUE;
      return FALSE;
    }

    patcher->pos += op;

    op = MIN (c.protected_bytes, count);
    count -= op;
    c.protected_bytes -= op;

    if (!append_protected_subsample (&patcher->dst, op)) {
      patcher->error = TRUE;
      return FALSE;
    }

    patcher->pos += op;

    if (count == 0)
      break;
  }

  return TRUE;
}

static GstBuffer *
make_subsample_buffer (GstSubsampleMap * subsample_map)
{
  const GstSubsample *begin;
  const GstSubsample *end;
  const GstSubsample *it;
  GstByteWriter bw;

  gst_byte_writer_init_with_size (&bw, subsample_map->subsample_count * 6,
      TRUE);

  begin = &subsample_map->subsamples[0];
  end = &subsample_map->subsamples[subsample_map->subsample_count];

  for (it = begin; it < end; ++it) {
    gst_byte_writer_put_uint16_be (&bw, it->clear_bytes);
    gst_byte_writer_put_uint32_be (&bw, it->protected_bytes);
  }

  return gst_byte_writer_reset_and_get_buffer (&bw);
}

static gsize
count_clear_subsample (GstSubsampleMap * subsample_map)
{
  const GstSubsample *begin;
  const GstSubsample *end;
  const GstSubsample *it;
  gsize count;

  count = 0;

  begin = &subsample_map->subsamples[0];
  end = &subsample_map->subsamples[subsample_map->subsample_count];

  for (it = begin; it < end; ++it)
    count += it->clear_bytes;

  return count;
}

gboolean
gst_subsample_patcher_commit (GstSubsamplePatcher * patcher, GstBuffer * buffer)
{
  GstProtectionMeta *prot_meta;
  gboolean encrypted;
  GstBuffer *buf;

  if (!patcher->init)
    return TRUE;

  if (patcher->error)
    return FALSE;

  prot_meta = gst_buffer_get_protection_meta (buffer);
  if (!prot_meta)
    return TRUE;

  if (!gst_structure_get_boolean (prot_meta->info, "encrypted", &encrypted))
    encrypted = FALSE;

  if (!encrypted)
    return TRUE;

  if (count_clear_subsample (&patcher->dst) == 0) {
    gst_structure_set (prot_meta->info, "subsample_count", G_TYPE_UINT, 0,
        NULL);
    gst_structure_remove_fields (prot_meta->info, "subsamples", NULL);
    return TRUE;
  }

  buf = make_subsample_buffer (&patcher->dst);

  gst_structure_set (prot_meta->info,
      "subsample_count", G_TYPE_UINT, patcher->dst.subsample_count,
      "subsamples", GST_TYPE_BUFFER, buf, NULL);

  gst_buffer_unref (buf);

  return TRUE;
}
