#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_GBM_PRIV
#define _GNU_SOURCE
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <gbm_priv.h>
#include <gst/allocators/gstdmabuf.h>
#include "wlvideoformat.h"
#endif

#include "wlgbmbufferbackend.h"

#ifdef HAVE_GBM_PRIV
GST_DEBUG_CATEGORY_EXTERN (gstwayland_debug);
#define GST_CAT_DEFAULT gstwayland_debug

typedef struct
{
  GMutex lock;
  GCond cond;
  struct wl_buffer *wbuf;
} ConstructBufferData;

static void
create_succeeded (void *data, struct gbm_buffer_params *params,
    struct wl_buffer *new_buffer)
{
  ConstructBufferData *d = data;

  g_mutex_lock (&d->lock);
  d->wbuf = new_buffer;
  gbm_buffer_params_destroy (params);
  g_cond_signal (&d->cond);
  g_mutex_unlock (&d->lock);
}

static void
create_failed (void *data, struct gbm_buffer_params *params)
{
  ConstructBufferData *d = data;

  g_mutex_lock (&d->lock);
  d->wbuf = NULL;
  gbm_buffer_params_destroy (params);
  g_cond_signal (&d->cond);
  g_mutex_unlock (&d->lock);
}

static const struct gbm_buffer_params_listener params_listener = {
  create_succeeded,
  create_failed
};

static int
create_metadata (const GstVideoInfo * info, gboolean secure, guint64 modifier,
    gboolean mastering_info_valid, const GstVideoMasteringDisplayInfo * minfo,
    gboolean cll_valid, const GstVideoContentLightLevel * cll)
{
  struct meta_data_t *meta;
  gsize size;
  gint i, fd;

  fd = memfd_create ("gbm_metadata", MFD_CLOEXEC | MFD_ALLOW_SEALING);
  if (fd < 0) {
    GST_ERROR ("failed to create metadata memfd");
    return -1;
  }

  size = sizeof (*meta);

  if (ftruncate (fd, size) < 0) {
    GST_ERROR ("failed to truncate metadata");
    goto fail;
  }

  if (fcntl (fd, F_ADD_SEALS, F_SEAL_SEAL | F_SEAL_SHRINK | F_SEAL_GROW) < 0)
    GST_WARNING ("failed to set seals on metadata");

  meta = mmap (NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
  if (meta == MAP_FAILED) {
    GST_ERROR ("failed to map metadata");
    goto fail;
  }

  memset (meta, 0, sizeof (*meta));
  meta->is_buffer_ubwc = !!(modifier & DRM_FORMAT_MOD_QCOM_COMPRESSED);
  meta->is_buffer_secure = secure;
  meta->color_info.colorPrimaries =
    gst_video_color_primaries_to_iso (info->colorimetry.primaries);
  meta->color_info.transfer =
      gst_video_transfer_function_to_iso (info->colorimetry.transfer);
  meta->color_info.matrixCoefficients =
    gst_video_color_matrix_to_iso (info->colorimetry.matrix);

  if (info->colorimetry.range == GST_VIDEO_COLOR_RANGE_0_255)
    meta->color_info.range = Range_Full;
  else
    meta->color_info.range = Range_Limited;

  if (cll_valid) {
    meta->color_info.contentLightLevel.lightLevelSEIEnabled = 1;
    meta->color_info.contentLightLevel.maxContentLightLevel =
      cll->max_content_light_level;
    meta->color_info.contentLightLevel.minPicAverageLightLevel =
      cll->max_frame_average_light_level;
  }

  if (mastering_info_valid) {
    meta->color_info.masteringDisplayInfo.colorVolumeSEIEnabled = 1;
    meta->color_info.masteringDisplayInfo.maxDisplayLuminance =
      minfo->max_display_mastering_luminance;
    meta->color_info.masteringDisplayInfo.minDisplayLuminance =
      minfo->min_display_mastering_luminance;
    meta->color_info.masteringDisplayInfo.primaries.whitePoint[0] =
      minfo->white_point.x;
    meta->color_info.masteringDisplayInfo.primaries.whitePoint[1] =
      minfo->white_point.y;
    for (i = 0; i < 3; i++) {
      meta->color_info.masteringDisplayInfo.primaries.rgbPrimaries[i][0] =
        minfo->display_primaries[i].x;
      meta->color_info.masteringDisplayInfo.primaries.rgbPrimaries[i][1] =
        minfo->display_primaries[i].y;
    }
  }

  return fd;

fail:
  close (fd);
  return -1;
}

struct wl_buffer *
gst_wl_gbm_buffer_backend_construct_wl_buffer (GstBuffer * buf,
    GstWlDisplay * display, const GstVideoInfo * info,
    gboolean secure, guint64 modifier,
    gboolean mastering_info_valid, const GstVideoMasteringDisplayInfo * minfo,
    gboolean cll_valid, const GstVideoContentLightLevel * cll)
{
  GstMemory *mem;
  int format;
  guint width, height;
  gint metadata_fd, fd;
  struct gbm_buffer_params *params;
  gint64 timeout;
  ConstructBufferData data;

  g_return_val_if_fail (gst_wl_display_check_format_for_dmabuf (display,
          GST_VIDEO_INFO_FORMAT (info), modifier), NULL);

  mem = gst_buffer_peek_memory (buf, 0);
  format = gst_video_format_to_wl_dmabuf_format (GST_VIDEO_INFO_FORMAT (info));

  g_cond_init (&data.cond);
  g_mutex_init (&data.lock);
  g_mutex_lock (&data.lock);

  width = GST_VIDEO_INFO_WIDTH (info);
  height = GST_VIDEO_INFO_HEIGHT (info);

  GST_DEBUG_OBJECT (display, "Creating gbm_buffer from DMABuf of size %"
      G_GSSIZE_FORMAT " (%d x %d), format %s:%lx", info->size, width, height,
      gst_wl_dmabuf_format_to_string (format), modifier);

  /* Hack to support TP10_UBWC */
  if (format == DRM_FORMAT_NV12 &&
      modifier == (DRM_FORMAT_MOD_QCOM_COMPRESSED |
                   DRM_FORMAT_MOD_QCOM_TIGHT | DRM_FORMAT_MOD_QCOM_DX))
    format = fourcc_code('Q', '1', '2', 'A');

  /* Creation and configuration of planes  */
  params = gbm_buffer_backend_create_params (display->gbm_buffer_backend);

  /* Request buffer creation */
  gbm_buffer_params_add_listener (params, &params_listener, &data);

  mem = gst_buffer_peek_memory (buf, 0);
  fd = gst_dmabuf_memory_get_fd (mem);

  metadata_fd = create_metadata (info, secure, modifier,
      mastering_info_valid, minfo, cll_valid, cll);
  if (metadata_fd < 0) {
    gbm_buffer_params_destroy (params);
    goto out;
  }

  gbm_buffer_params_create (params, fd, metadata_fd, width, height, format, 0);
  close (metadata_fd);

  /* Wait for the request answer */
  wl_display_flush (display->display);
  data.wbuf = (gpointer) 0x1;
  timeout = g_get_monotonic_time () + G_TIME_SPAN_SECOND;
  while (data.wbuf == (gpointer) 0x1) {
    if (!g_cond_wait_until (&data.cond, &data.lock, timeout)) {
      GST_ERROR_OBJECT (mem->allocator, "gbm_buffer_params time out");
      gbm_buffer_params_destroy (params);
      data.wbuf = NULL;
    }
  }

out:
  if (!data.wbuf) {
    GST_ERROR_OBJECT (mem->allocator, "can't create gbm buffer");
  } else {
    GST_DEBUG_OBJECT (mem->allocator, "created gbm wl_buffer (%p):"
        "%dx%d, fmt=%.4s", data.wbuf, width, height, (char *) &format);
  }

  g_mutex_unlock (&data.lock);
  g_mutex_clear (&data.lock);
  g_cond_clear (&data.cond);

  return data.wbuf;
}

#else
struct wl_buffer *
gst_wl_gbm_buffer_backend_construct_wl_buffer (GstBuffer * buf,
    GstWlDisplay * display, const GstVideoInfo * info, gboolean secure,
    guint64 modifier, gboolean mastering_info_valid,
    const GstVideoMasteringDisplayInfo * minfo, gboolean cll_valid,
    const GstVideoContentLightLevel * cll)
{
  return NULL;
}
#endif
